﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Text;
using System.Windows.Forms;

using Peak.Can.IsoTp;
using TPCANTPHandle = System.UInt16;
using System.Runtime.InteropServices;

namespace PcanIsoTpExample
{
    /// <summary>
    /// Main application that demonstrates ISO-TP API features.
    /// </summary>
    public partial class FormMain : Form
    {
        #region Delegates
        /// <summary>
        /// Declare a delegate type to read CANTP messages
        /// </summary>
        public delegate void ReadDelegateHandler();
        /// <summary>
        /// Declare a delegate type to handle failure while reading CANTP messages
        /// </summary>
        public delegate void ReadFailedDelegateHandler(TPCANTPStatus sts);
        #endregion

        #region Members
        /// <summary>
        /// Handles of all currently available PCAN-Hardwares.
        /// </summary>
        private TPCANTPHandle[] m_pctpHandles;
        /// <summary>
        /// Connected PCAN ISO-TP Handle (PCANTP_NONEBUS if not connected).
        /// </summary>
        private TPCANTPHandle m_pctpHandle;
        /// <summary>
        /// DialogBox that handles mapping.
        /// </summary>
        private FormMapping m_dlgMapping;
        /// <summary>
        /// Bit rate text
        /// </summary>
        private string m_textCanFd;

        /// <summary>
        /// List of configured PCAN-ISO-TP mappings.
        /// </summary>
        private List<MappingStatus> m_mappings;
        /// <summary>
        /// Collection of MessageStatus used to display each received ISO-TP messages.
        /// </summary>
        private System.Collections.ArrayList m_receiveMsgs;
        /// <summary>
        /// "Receive CANTP message" Event.
        /// </summary>
        private System.Threading.AutoResetEvent m_receiveEvent;
        /// <summary>
        /// Read Delegate to read CANTP messages.
        /// </summary>
        private ReadDelegateHandler m_readDelegate;
        /// <summary>
        /// Read Delegate to read CANTP messages.
        /// </summary>
        private ReadFailedDelegateHandler m_readFailedDelegate;
        /// <summary>
        /// Thread used to read CANTP messages via events.
        /// </summary>
        private System.Threading.Thread m_readThread;

        #endregion

        /// <summary>
        /// Default constructor
        /// </summary>
        public FormMain()
        {
            InitializeComponent();
            // Asserts PCAN-ISO-TP library is available.
            try
            {
                StringBuilder buf = new StringBuilder(256);
                TPCANTPStatus sts;
                // Get API version
                sts = CanTpApi.GetValue(CanTpApi.PCANTP_NONEBUS, TPCANTPParameter.PCANTP_PARAM_API_VERSION, buf, (uint)buf.Capacity);
                CanTpUtils.checkCanTpStatus(CanTpApi.PCANTP_NONEBUS, sts, buf, TPCANTPParameter.PCANTP_PARAM_API_VERSION);
            }
            catch (DllNotFoundException)
            {
                MessageBox.Show("Unable to find one of required PCAN libraries: PCANBasic.dll or PCAN-ISO-TP.dll.", "Error!",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                Environment.Exit(-1);
            }
            // Creates an array with all possible PCAN-ISO-TP Channels.
            m_pctpHandles = new TPCANTPHandle[]
            {
                CanTpApi.PCANTP_ISABUS1,
                CanTpApi.PCANTP_ISABUS2,
                CanTpApi.PCANTP_ISABUS3,
                CanTpApi.PCANTP_ISABUS4,
                CanTpApi.PCANTP_ISABUS5,
                CanTpApi.PCANTP_ISABUS6,
                CanTpApi.PCANTP_ISABUS7,
                CanTpApi.PCANTP_ISABUS8,
                CanTpApi.PCANTP_DNGBUS1,
                CanTpApi.PCANTP_PCIBUS1,
                CanTpApi.PCANTP_PCIBUS2,
                CanTpApi.PCANTP_PCIBUS3,
                CanTpApi.PCANTP_PCIBUS4,
                CanTpApi.PCANTP_PCIBUS5,
                CanTpApi.PCANTP_PCIBUS6,
                CanTpApi.PCANTP_PCIBUS7,
                CanTpApi.PCANTP_PCIBUS8,
                CanTpApi.PCANTP_PCIBUS9,
                CanTpApi.PCANTP_PCIBUS10,
                CanTpApi.PCANTP_PCIBUS11,
                CanTpApi.PCANTP_PCIBUS12,
                CanTpApi.PCANTP_PCIBUS13,
                CanTpApi.PCANTP_PCIBUS14,
                CanTpApi.PCANTP_PCIBUS15,
                CanTpApi.PCANTP_PCIBUS16,
                CanTpApi.PCANTP_USBBUS1,
                CanTpApi.PCANTP_USBBUS2,
                CanTpApi.PCANTP_USBBUS3,
                CanTpApi.PCANTP_USBBUS4,
                CanTpApi.PCANTP_USBBUS5,
                CanTpApi.PCANTP_USBBUS6,
                CanTpApi.PCANTP_USBBUS7,
                CanTpApi.PCANTP_USBBUS8,
                CanTpApi.PCANTP_USBBUS9,
                CanTpApi.PCANTP_USBBUS10,
                CanTpApi.PCANTP_USBBUS11,
                CanTpApi.PCANTP_USBBUS12,
                CanTpApi.PCANTP_USBBUS13,
                CanTpApi.PCANTP_USBBUS14,
                CanTpApi.PCANTP_USBBUS15,
                CanTpApi.PCANTP_USBBUS16,
                CanTpApi.PCANTP_PCCBUS1,
                CanTpApi.PCANTP_PCCBUS2,
                CanTpApi.PCANTP_LANBUS1,
                CanTpApi.PCANTP_LANBUS2,
                CanTpApi.PCANTP_LANBUS3,
                CanTpApi.PCANTP_LANBUS4,
                CanTpApi.PCANTP_LANBUS5,
                CanTpApi.PCANTP_LANBUS6,
                CanTpApi.PCANTP_LANBUS7,
                CanTpApi.PCANTP_LANBUS8,
                CanTpApi.PCANTP_LANBUS9,
                CanTpApi.PCANTP_LANBUS10,
                CanTpApi.PCANTP_LANBUS11,
                CanTpApi.PCANTP_LANBUS12,
                CanTpApi.PCANTP_LANBUS13,
                CanTpApi.PCANTP_LANBUS14,
                CanTpApi.PCANTP_LANBUS15,
                CanTpApi.PCANTP_LANBUS16,
            };
            // Initializes array of received ISO-TP messages
            m_receiveMsgs = new System.Collections.ArrayList();
            // Initializes list of configured mappings
            m_mappings = new List<MappingStatus>();
            // Creates the delegate used to message reading
            m_readDelegate = new ReadDelegateHandler(readMessages);
            // Creates the delegate used when reading-thread fails
            m_readFailedDelegate = new ReadFailedDelegateHandler(readThreadFailed);
            // Creates the event used to signalize incomming messages 
            m_receiveEvent = new System.Threading.AutoResetEvent(false);

            // Initializes UI components
            initializeUiConnection();
            initializeUiTabMessages();
            initializeUiTabParameters();
            // Initializes the "mapping" dialog once and for all
            m_dlgMapping = new FormMapping();
            m_textCanFd = "f_clock_mhz=20, nom_brp=5, nom_tseg1=2, nom_tseg2=1, nom_sjw=1, data_brp=2, data_tseg1=3, data_tseg2=1, data_sjw=1";
            textBoxCanFdBitrate.Text = m_textCanFd;
            checkBoxCanFd_CheckedChanged(this, null); // initialization of UI components for FD messages
        }

        #region Main Form event handlers
        /// <summary>
        /// Event handler called when main window is first shown.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void FormMain_Shown(object sender, EventArgs e)
        {
            // UI Components are all initialized, refresh mappings' tab.
            mappingsChanged();
            // Set connection status to false to update all UI's components
            setConnectionStatus(false);
        }

        /// <summary>
        /// Event handler called when main window is closed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void FormMain_FormClosed(object sender, FormClosedEventArgs e)
        {
            // Disconnect all PCAN-ISO-TP connections.
            disconnect();
        }
        #endregion

        #region Connection > Event Handlers
        /// <summary>
        /// Event handler called when button "Hardware Refresh" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonHwRefresh_Click(object sender, EventArgs e)
        {
            TPCANTPStatus sts;
            ComboBoxItem cbItem;                // an item of the "channel" comboBox
            TPCANTPHandle pctpHandle;           // temporary PCAN ISO-TP handle
            TPCANTPHandle pctpHandleSelected;   // store the currently selected channel in the "channel" comboBox
            UInt32 iBuffer;                     // temporary buffer to retrieve the condition of a channel

            // Get selected item first (to set it back later).
            if (comboBoxChannel.SelectedItem != null &&
                comboBoxChannel.SelectedItem is ComboBoxItem)
                pctpHandleSelected = (TPCANTPHandle)((ComboBoxItem)comboBoxChannel.SelectedItem).Data;
            else
                pctpHandleSelected = CanTpApi.PCANTP_NONEBUS;

            // Clears the "Channel" comboBox and fill it again with 
            //  all the PCAN-ISO-TP handles (non-plug'n'play hardware and
            //  the detected Plug'n'Play hardware).
            comboBoxChannel.Items.Clear();
            // Loop through all known PCAN-ISO-TP handles.
            for (int i = 0; i < m_pctpHandles.Length; i++)
            {
                pctpHandle = m_pctpHandles[i];
                // Initializes an item for the comboBox.
                cbItem = new ComboBoxItem(CanTpUtils.GetChannelName(pctpHandle), pctpHandle);
                // Automatically add non-plug'n'play handles.
                if (pctpHandle <= CanTpApi.PCANTP_DNGBUS1)
                    comboBoxChannel.Items.Add(cbItem);
                else
                {
                    // Check the availalility of plug'n'play handles.
                    sts = CanTpApi.GetValue(pctpHandle, TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION, out iBuffer, sizeof(UInt32));
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK &&
                        ((iBuffer & CanTpApi.PCANTP_CHANNEL_AVAILABLE) == CanTpApi.PCANTP_CHANNEL_AVAILABLE))
                    {
                        // Channel is available, add it to the comboBox.
                        comboBoxChannel.Items.Add(cbItem);
                    }
                }
                // Select the handle if it was the previously selected one.
                if (pctpHandle == pctpHandleSelected)
                    comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1;
            }
        }
        /// <summary>
        /// Event handler called when button "Initialize" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonInit_Click(object sender, EventArgs e)
        {
            TPCANTPHandle pctpHandle;
            TPCANTPBaudrate baudrate;
            TPCANTPHWType hwType;
            uint ioPort;
            byte interrupt;

            if (comboBoxChannel.SelectedItem == null)
                return;
            // Gets the selected ISO-TP channel.
            pctpHandle = (TPCANTPHandle)((ComboBoxItem)comboBoxChannel.SelectedItem).Data;
            // Gets the selected baudrate.
            baudrate = (TPCANTPBaudrate)((ComboBoxItem)comboBoxBaudrate.SelectedItem).Data;
            // Read the HarwareType, IO port and Interrupt only 
            //  if the selected device is not plug'n'play.
            hwType = 0;
            ioPort = 0;
            interrupt = 0;
            if (pctpHandle <= CanTpApi.PCANTP_DNGBUS1)
            {
                if (comboBoxHwType.SelectedIndex >= 0)
                    hwType = (TPCANTPHWType)((ComboBoxItem)comboBoxHwType.SelectedItem).Data;
                if (comboBoxIoPort.SelectedIndex >= 0)
                    ioPort = (uint)((ComboBoxItem)comboBoxIoPort.SelectedItem).Data;
                if (comboBoxInterrupt.SelectedIndex >= 0)
                    interrupt = (byte)((ComboBoxItem)comboBoxInterrupt.SelectedItem).Data;
            }
            // Connects to the selected PCAN-ISO-TP channel.
            connect(pctpHandle, baudrate, hwType, ioPort, interrupt);
        }
        /// <summary>
        /// Event handler called when button "Release" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonRelease_Click(object sender, EventArgs e)
        {
            // Disconnects the connected PCAN-ISO-TP channel.
            disconnect();
        }

        /// <summary>
        /// Event handler called when selection of comboBox "channel handles" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxChannel_SelectedIndexChanged(object sender, EventArgs e)
        {
            TPCANTPHandle pctpHandle;
            bool isNotPnp;

            // Enables "Initialize" button if a channel is selected.
            buttonInit.Enabled = (comboBoxChannel.SelectedItem != null);
            // Aborts if no channel is selected.
            if (comboBoxChannel.SelectedItem == null)
                return;
            // Gets the handle from the selected item.
            pctpHandle = (TPCANTPHandle)((ComboBoxItem)comboBoxChannel.SelectedItem).Data;
            // Determines if the handle belong to a non-plug'n'play hardware.
            isNotPnp = (pctpHandle <= CanTpApi.PCANTP_DNGBUS1);
            // Activates/deactivates configuration controls according to the kind of hardware.
            comboBoxHwType.Enabled = isNotPnp;
            comboBoxIoPort.Enabled = isNotPnp;
            comboBoxInterrupt.Enabled = isNotPnp;
        }
        #endregion

        #region Parameters > Event Handlers
        /// <summary>
        /// Event handler called when button "Parameters > Get" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamGet_Click(object sender, EventArgs e)
        {
            TPCANTPParameter param;
            TPCANTPStatus sts;
            uint iBuf;
            StringBuilder strBuf;
            TPCANTPHandle canHandle;

            // Checks selection is valid.
            if (comboBoxParameter.SelectedItem == null || !(comboBoxParameter.SelectedItem is ComboBoxItem))
                return;
            // Gets selected parameter
            param = (TPCANTPParameter)((ComboBoxItem)comboBoxParameter.SelectedItem).Data;
            iBuf = 0;
            switch (param)
            {
                // Retrieves the API version.
                case TPCANTPParameter.PCANTP_PARAM_API_VERSION:
                    strBuf = new StringBuilder(256);
                    sts = CanTpApi.GetValue(m_pctpHandle, param, strBuf, (uint)strBuf.Capacity);
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, strBuf, param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage("ISO-TP API version:" + strBuf);
                    break;
                // Reads the ISO-TP BlockSize (BS) parameter (number of Consecutive Frame between Flow Control frames).
                case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("ISO-TP Block Size parameter is {0} ({0:X2}h).", iBuf));
                    break;
                // Reads if CAN DATA Padding is enabled (if enabled, all segmented CAN frames have a DLC of 8).
                case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("CAN Data Padding is {0} ({1}).",
                            (iBuf == CanTpApi.PCANTP_CAN_DATA_PADDING_NONE) ? "disabled" : "enabled", iBuf));
                    break;
                // Reads the channel's state.
                case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION:
                    // Gets the selected ISO-TP channel.
                    if (comboBoxChannel.SelectedItem is ComboBoxItem)
                        canHandle = (TPCANTPHandle)((ComboBoxItem)comboBoxChannel.SelectedItem).Data;
                    else
                        canHandle = m_pctpHandle;
                    sts = CanTpApi.GetValue(canHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                    {
                        switch (iBuf)
                        {
                            case CanTpApi.PCANTP_CHANNEL_AVAILABLE:
                                includeTextMessage(String.Format("Channel {0:X2}h is available ({1}).", canHandle, iBuf));
                                break;
                            case CanTpApi.PCANTP_CHANNEL_UNAVAILABLE:
                                includeTextMessage(String.Format("Channel {0:X2}h is unavailable ({1}).", canHandle, iBuf));
                                break;
                            case CanTpApi.PCANTP_CHANNEL_OCCUPIED:
                                includeTextMessage(String.Format("Channel {0:X2}h is occupied ({1}).", canHandle, iBuf));
                                break;
                            default:
                                includeTextMessage(String.Format("Channel {0:X2}h is in an unknown state ({1}).", canHandle, iBuf));
                                break;
                        }
                    }
                    break;
                // Reads if Debug mode is enabled (if enabled, CAN Tx/Rx frames are echoed on console output).
                case TPCANTPParameter.PCANTP_PARAM_DEBUG:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Debug mode 'CAN frames on console output' is {0} ({1}).",
                            (iBuf == CanTpApi.PCANTP_DEBUG_NONE) ? "disabled" : "enabled", iBuf));
                    break;
                // Reads if Message Pending Mode is enabled 
                //  (if enabled, CAN ISO-TP messages with type PCANTP_MESSAGE_INDICATION 
                //  can be read when the first frame of a segmented message is received).
                case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Notifications of incoming CAN-ISO-TP messages are {0} ({1}).",
                            (iBuf == CanTpApi.PCANTP_MSG_PENDING_HIDE) ? "disabled" : "enabled", iBuf));
                    break;
                // Reads the padding value used when "CAN DATA Padding" is enabled.
                case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Value for 'CAN Data Padding' is {0:X2}h ({0}).", iBuf));
                    break;
                // Reads the ISO-TP default priority value for normal fixed, mixed and enhanced addressing
                case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("The default priority value for normal fixed, mixed and enhanced addressing is {0:X2}h ({0}).", iBuf));
                    break;
                // Reads the ISO-TP default DLC to use when transmitting messages with CAN FD
                case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("The default DLC to use when transmitting messages with CAN FD is {0:X2}h ({0}).", iBuf));
                    break;
                // Reads the receive-event handle (if receive-event is disabled, 0 is returned).
                case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Receive event is {0} (event handle is {1:X}h).",
                            (iBuf == 0) ? "disabled" : "enabled", iBuf));
                    break;
                // Reads the ISO-TP SeparationTime (STmin) parameter 
                //  (minimum time to wait between Consecutive Frames transmission).
                case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                    {
                        if (iBuf < 0x80)
                            includeTextMessage(String.Format("ISO-TP Separation Time (STmin) is {0}ms ({0:X2}h).", iBuf));
                        else if (iBuf > 0xF0 && iBuf < 0xFA)
                            includeTextMessage(String.Format("ISO-TP Separation Time (STmin) is {0}µs ({0:X2}h).", (iBuf - 0xF0) * 100));
                        else
                            includeTextMessage(String.Format("ISO-TP Separation Time (STmin) uses a reserved value of ISO 15765 ({0:X2}h).", iBuf));
                    }
                    break;
                // Reads the ISO-TP FC.Wait frame transmissions (N_WFTMax) parameter 
                //  (maximum number of consecutive Flow Control frames with the Wait status after which transmission is aborted).
                case TPCANTPParameter.PCANTP_PARAM_WFT_MAX:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Maximum number of FC.Wait frame transmissions (N_WFTMax) is {0}.", iBuf));
                    break;
                // Generic reading request of a parameter (should not occur).
                default:
                    sts = CanTpApi.GetValue(m_pctpHandle, param, out iBuf, sizeof(uint));
                    CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (byte)iBuf, (int)param);
                    if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                        includeTextMessage(String.Format("Unknown parameter '{0}' is {1:X2}h", param, iBuf));
                    break;
            }
        }
        /// <summary>
        /// Event handler called when button "Parameters > Set" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamSet_Click(object sender, EventArgs e)
        {
            TPCANTPParameter param;
            TPCANTPStatus sts;
            uint iBuf = 0;

            // Checks selection is valid.
            if (comboBoxParameter.SelectedItem == null || !(comboBoxParameter.SelectedItem is ComboBoxItem))
                return;
            // Sets selected parameter
            param = (TPCANTPParameter)((ComboBoxItem)comboBoxParameter.SelectedItem).Data;
            switch (param)
            {
                // Sets the ISO-TP BlockSize (BS) parameter (number of Consecutive Frame between Flow Control frames).
                case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets if CAN DATA Padding is enabled (if enabled, all segmented CAN frames have a DLC of 8).
                case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING:
                    iBuf = radioButtonParamActive.Checked ? CanTpApi.PCANTP_CAN_DATA_PADDING_ON : CanTpApi.PCANTP_CAN_DATA_PADDING_NONE;
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets if Debug mode is enabled (if enabled, CAN Tx/Rx frames are echoed on console output).
                case TPCANTPParameter.PCANTP_PARAM_DEBUG:
                    iBuf = radioButtonParamActive.Checked ? CanTpApi.PCANTP_DEBUG_CAN : CanTpApi.PCANTP_DEBUG_NONE;
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets if Message Pending Mode is enabled 
                //  (if enabled, CAN ISO-TP messages with type PCANTP_MESSAGE_INDICATION 
                //  can be read when the first frame of a segmented message is received).
                case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING:
                    iBuf = radioButtonParamActive.Checked ? CanTpApi.PCANTP_MSG_PENDING_SHOW : CanTpApi.PCANTP_MSG_PENDING_HIDE;
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets the padding value used when "CAN DATA Padding" is enabled.
                case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets the priority value used when "J1939 Priority" is enabled.
                case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets the DLC value used when "Data Length Code (DLC)" is enabled.
                case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets the ISO-TP SeparationTime (STmin) parameter 
                //  (minimum time to wait between Consecutive Frames transmission).
                case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Sets the ISO-TP FC.Wait frame transmissions (N_WFTMax) parameter 
                //  (maximum number of consecutive Flow Control frames with the Wait status after which transmission is aborted).
                case TPCANTPParameter.PCANTP_PARAM_WFT_MAX:
                    iBuf = Convert.ToUInt32(numericUpDownParamValue.Value);
                    sts = CanTpApi.SetValue(m_pctpHandle, param, ref iBuf, sizeof(uint));
                    break;
                // Unhandled parameter (should not occur).
                default:
                    includeTextMessage(String.Format("Unknown parameter '{0}', setting aborted.", param));
                    sts = TPCANTPStatus.PCANTP_ERROR_WRONG_PARAM;
                    break;
            }
            CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (int)param, (int)iBuf);
            // Outputs the new parameter value.
            buttonParamGet_Click(this, EventArgs.Empty);
        }
        /// <summary>
        /// Event handler called when button "Parameters > Clear Information" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamInfoClear_Click(object sender, EventArgs e)
        {
            // Clears the information of the Information List-Box 
            listBoxParamInfo.Items.Clear();
        }
        /// <summary>
        /// Event handler called when button "Parameters > Reset" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamReset_Click(object sender, EventArgs e)
        {
            TPCANTPStatus sts;

            // Resets the receive and transmit queues of a PCAN Channel.
            sts = CanTpApi.Reset(m_pctpHandle);
            CanTpUtils.checkCanTpStatus(m_pctpHandle, sts);
            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                includeTextMessage("Receive and transmit queues successfully reset.");
        }
        /// <summary>
        /// Event handler called when button "Parameters > Status" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamStatus_Click(object sender, EventArgs e)
        {
            TPCANTPStatus stsResult;
            string errorName;

            // Gets the current BUS status of a PCAN Channel.
            stsResult = CanTpApi.GetStatus(m_pctpHandle);
            // Formats on status
            switch (stsResult)
            {
                case TPCANTPStatus.PCANTP_ERROR_NOT_INITIALIZED:
                    errorName = "PCANTP_ERROR_NOT_INITIALIZED";
                    break;
                case TPCANTPStatus.PCANTP_ERROR_BUSLIGHT:
                    errorName = "PCANTP_ERROR_BUSLIGHT";
                    break;
                case TPCANTPStatus.PCANTP_ERROR_BUSHEAVY: // TPCANTPStatus.PCANTP_ERROR_BUSWARNING
                    errorName = "PCANTP_ERROR_BUSHEAVY";
                    break;
                case TPCANTPStatus.PCANTP_ERROR_BUSOFF:
                    errorName = "PCANTP_ERROR_BUSOFF";
                    break;
                case TPCANTPStatus.PCANTP_ERROR_OK:
                    errorName = "PCANTP_ERROR_OK";
                    break;
                default:
                    errorName = "See Documentation";
                    break;
            }
            // Displays Message
            includeTextMessage(String.Format("Channel status: {0} ({1:X}h)", errorName, stsResult));
        }
        /// <summary>
        /// Event handler called when button "Parameters > Version" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonParamVersion_Click(object sender, EventArgs e)
        {

            TPCANTPStatus sts;
            StringBuilder strTemp;
            //Gets the vesion of the ISO-TP API
            strTemp = new StringBuilder(256);
            sts = CanTpApi.GetValue(CanTpApi.PCANTP_NONEBUS,
                TPCANTPParameter.PCANTP_PARAM_API_VERSION, strTemp, (uint)strTemp.Capacity);
            CanTpUtils.checkCanTpStatus(CanTpApi.PCANTP_NONEBUS, sts, strTemp, TPCANTPParameter.PCANTP_PARAM_API_VERSION);
            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
            {
                includeTextMessage("API Version: " + strTemp.ToString());
            }
            //CCANExamplesValidation.displayValidatioMsg(sts, 0, TPCANTPParameter.PCANTP_PARAM_API_VERSION, (uint)strTemp.Length, strTemp.ToString(), CCANExamplesValidation.CallerName());
        }

        /// <summary>
        /// Event handler called when the selection of the comboBox "Parameters" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxParameter_SelectedIndexChanged(object sender, EventArgs e)
        {
            TPCANTPParameter param;
            bool isRadioEnabled, isNudEnabled, isSetEnabled, isGetEnabled;
            uint nudMaxValue, nudMinValue, nudDefaultValue;

            nudMinValue = 0x00;
            nudMaxValue = 0xFF;
            nudDefaultValue = 0x00;

            isRadioEnabled = false;
            isNudEnabled = false;
            isSetEnabled = false;
            isGetEnabled = false;
            if (comboBoxParameter.SelectedItem != null &&
                comboBoxParameter.SelectedItem is ComboBoxItem)
            {
                isGetEnabled = isConnected();
                // Activates/deactivates controls according to the selected parameter 
                param = (TPCANTPParameter)((ComboBoxItem)comboBoxParameter.SelectedItem).Data;
                switch (param)
                {
                    case TPCANTPParameter.PCANTP_PARAM_API_VERSION:
                        isGetEnabled = true;
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE:
                        isNudEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING:
                        isRadioEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION:
                        isGetEnabled = true;
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_DEBUG:
                        isRadioEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING:
                        isRadioEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE:
                        isNudEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY:
                        isNudEnabled = true;
                        nudMaxValue = 0x07;
                        nudDefaultValue = 0x06;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL:
                        isNudEnabled = true;
                        nudMaxValue = 0x0F;
                        nudMinValue = 0x08;
                        nudDefaultValue = 0x08;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT:
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME:
                        isNudEnabled = true;
                        isSetEnabled = isConnected();
                        break;
                    case TPCANTPParameter.PCANTP_PARAM_WFT_MAX:
                        isGetEnabled = true;
                        isNudEnabled = true;
                        isSetEnabled = true;
                        nudMaxValue = 0xFFFF;
                        break;
                }
            }
            // Activates/deactivates controls
            buttonParamGet.Enabled = isGetEnabled;
            buttonParamSet.Enabled = isSetEnabled;
            radioButtonParamActive.Enabled = isRadioEnabled;
            radioButtonParamInactive.Enabled = isRadioEnabled;
            numericUpDownParamValue.Enabled = isNudEnabled;
            numericUpDownParamValue.Maximum = nudMaxValue;
            numericUpDownParamValue.Minimum = nudMinValue;
            numericUpDownParamValue.Value = nudDefaultValue;
        }
        #endregion

        #region Mapping > Event Handlers
        /// <summary>
        /// Event handler called when button "Add Mapping" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMappingAdd_Click(object sender, EventArgs e)
        {
            // Prepares dialogBox's UI components with a default mapping.
            m_dlgMapping.initializeMapping(null);
            // Shows dialog
            if (m_dlgMapping.ShowDialog() == DialogResult.OK)
            {
                // Clones the mapping from the dialogBox and adds it to the API.
                mappingsAdd(m_dlgMapping.m_mapping.Clone());
                // Refreshes UI dealing with mappings.
                mappingsChanged();
            }
        }

        /// <summary>
        /// Event handler called when button "Delete Mapping" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMappingDel_Click(object sender, EventArgs e)
        {
            // Asserts the selection is valid
            if (listViewMappings.SelectedItems.Count > 0 &&
                listViewMappings.SelectedItems[0].Tag is MappingStatus)
            {
                MappingStatus mapping;
                TPCANTPStatus sts;

                // Gets the selected mapping.
                mapping = (MappingStatus)listViewMappings.SelectedItems[0].Tag;
                // Prevent deletion of "automatic mapping" 
                //  (those mappings do not exist in the ISO-TP API, they are displayed
                //  as a tutorial purpose)
                if (mapping.IsAutomatic)
                    return;
                // Removes the mapping from the API (via the CAN ID that defines it uniquely)
                sts = CanTpApi.RemoveMapping(m_pctpHandle, mapping.m_canId);
                CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, (int)mapping.m_canId);
                if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
                {
                    // Removes the mapping from the internal list of mappings
                    m_mappings.Remove(mapping);
                    // Removes the corresponding listViewItem 
                    listViewMappings.Items.Remove(listViewMappings.SelectedItems[0]);
                }
            }
        }

        /// <summary>
        /// Event handler called when button "Mapping example" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMappingSample_Click(object sender, EventArgs e)
        {
            MappingStatus mapping;
            byte clientAddr;    // Tester Client address
            byte ecuAddr;       // ECU address
            byte funcAddr;      // Functional address
            uint canIdCli2Ecu;  // CAN ID used to communicate from Client to ECU
            uint canIdEcu2Cli;  // CAN ID used to communicate from ECU to Client
            uint canIdCliFunc;  // CAN ID used to communicate from Client to any ECUs

            // The sample defines 11 bits CAN ID, normal format addressing mappings 
            //  (diagnostic message will be mandatory since normal format addressing is used)
            clientAddr = 0xF1;
            ecuAddr = 0x13;
            funcAddr = 0x20;
            canIdCli2Ecu = 0xA1;
            canIdEcu2Cli = 0xA2;
            canIdCliFunc = 0xA3;
            // Defines a mapping to allow communication from client to ECU (a.k.a request).
            mapping = new MappingStatus(canIdCli2Ecu, canIdEcu2Cli,
                TPCANTPIdType.PCANTP_ID_CAN_11BIT,
                TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
                TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL,
                clientAddr, ecuAddr, 0x00);
            mappingsAdd(mapping);
            // Defines a mapping to allow communication from ECU to client (a.k.a response).
            mapping = new MappingStatus(canIdEcu2Cli, canIdCli2Ecu,
                TPCANTPIdType.PCANTP_ID_CAN_11BIT,
                TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
                TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL,
                ecuAddr, clientAddr, 0x00);
            mappingsAdd(mapping);
            // define mapping to allow communication from client to any ECUs (a.k.a functional request).
            mapping = new MappingStatus(canIdCliFunc, CanTpApi.CAN_ID_NO_MAPPING,
                TPCANTPIdType.PCANTP_ID_CAN_11BIT,
                TPCANTPFormatType.PCANTP_FORMAT_NORMAL,
                TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL,
                clientAddr, funcAddr, 0x00);
            mappingsAdd(mapping);
            // Refreshes all UIs dealing with mappings.
            mappingsChanged();
        }

        /// <summary>
        /// Event handler called when selection of listview "mapping" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void listViewMappings_SelectedIndexChanged(object sender, EventArgs e)
        {
            bool enabled;

            // Enable/disable "remove mapping" button based on listView's selection 
            enabled = (listViewMappings.SelectedItems.Count > 0);
            if (enabled)
            {
                // Also do not allow removal of "automatic mapping" (since its a cosmetic display).
                MappingStatus mapping = (MappingStatus)listViewMappings.SelectedItems[0].Tag;
                if (mapping != null && mapping.IsAutomatic)
                    enabled = false;
            }
            // Updates button status.
            buttonMappingDel.Enabled = enabled;
        }
        #endregion

        #region Messages > Read Event Handlers
        /// <summary>
        /// Event handler called when button "Clear message" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMsgClear_Click(object sender, EventArgs e)
        {
            TPCANTPStatus sts;

            // Clears ISO-TP Rx/Tx queues only if channel is connected.
            if (isConnected())
            {
                sts = CanTpApi.Reset(m_pctpHandle);
                CanTpUtils.checkCanTpStatus(m_pctpHandle, sts);
            }
            // Clears "received messages" listview and internal list of received messages.
            lock (m_receiveMsgs.SyncRoot)
            {
                m_receiveMsgs.Clear();
                listViewMsgs.Items.Clear();
            }
        }
        /// <summary>
        /// Event handler called when button "Read message" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMsgRead_Click(object sender, EventArgs e)
        {
            // Reads one ISO-TP message.
            readMessage();
        }

        /// <summary>
        /// Event handler called when the value of the checkBox "Show Timestamp as period" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void checkBoxMsgShowPeriod_CheckedChanged(object sender, EventArgs e)
        {
            // According to the check-value of this checkbox,
            //  the received time of a messages will be interpreted as a
            //  period (time between the two last messages) or as time-stamp
            //  (the elapsed time since windows was started).
            lock (m_receiveMsgs.SyncRoot)
            {
                foreach (MessageStatus msg in m_receiveMsgs)
                    msg.ShowingPeriod = checkBoxMsgShowPeriod.Checked;
            }
        }

        /// <summary>
        /// Event handler called when radioButton "Reading Type" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void radioButtonMsgRead_CheckedChanged(object sender, EventArgs e)
        {
            // Terminates "read" thread if it exists.
            if (m_readThread != null)
            {
                m_readThread.Abort();
                m_readThread.Join();
                m_readThread = null;
            }
            // Disables "read" timer.
            timerRead.Enabled = false;
            // Updates enable state of manual read button.
            buttonMsgRead.Enabled = isConnected() && radioButtonMsgManual.Checked;
            // Updates timer that diplays new messages.
            timerDisplay.Enabled = isConnected();
            // Aborts function if no channel is connected.
            if (!isConnected())
                return;
            // According to the kind of reading, either a timer, a thread or a button is enabled.
            if (radioButtonMsgTimer.Checked)
            {
                // Enables Timer.
                timerRead.Enabled = isConnected();
            }
            else if (radioButtonMsgEvent.Checked)
            {
                // Creates and starts the tread to read CAN ISO-TP Message using PCANTP_PARAM_RECEIVE_EVENT.
                System.Threading.ThreadStart threadDelegate = new System.Threading.ThreadStart(pctpReadThread);
                m_readThread = new System.Threading.Thread(threadDelegate);
                m_readThread.IsBackground = true;
                m_readThread.Start();
            }
        }

        /// <summary>
        /// Event handler called periodically to update the display of received messages.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void timerDisplay_Tick(object sender, EventArgs e)
        {
            displayMessages();
        }
        /// <summary>
        /// Event handler called periodically to read new ISO-TP messages.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void timerRead_Tick(object sender, EventArgs e)
        {
            // Try to read new messages.
            readMessages();
        }
        #endregion

        #region Messages > Write Event Handlers        
        /// <summary>
        /// Event handler called when the button "Fill message's data" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMsgDataFill_Click(object sender, EventArgs e)
        {
            StringBuilder buf;
            char[] byteStr;

            // Generates a 4095-bytes String (each byte is increased by 1)
            buf = new StringBuilder(4095 * 2);
            for (int i = 1, count = 0; count < buf.Capacity; i++, count += 2)
            {
                byteStr = String.Format("{0:X2}", i % 256, 16).ToCharArray();
                buf.Append(byteStr[0]);
                buf.Append(byteStr[1]);
            }
            // Sets the data according to its length
            textBoxMsgData.Text = buf.ToString().Substring(0, textBoxMsgData.MaxLength).ToUpper();
        }
        /// <summary>
        /// Event handler called when the button "Write message" is clicked.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void buttonMsgWrite_Click(object sender, EventArgs e)
        {
            TPCANTPMsg msg;
            MappingStatus mapping;
            TPCANTPStatus sts;
            string strByte;

            // Asserts a mapping to write is selected.
            if (comboBoxMsgMapping.SelectedItem == null)
                return;
            // Gets the selected mapping and asserts it is valid.
            mapping = (MappingStatus)(((ComboBoxItem)comboBoxMsgMapping.SelectedItem).Data);
            if (mapping == null)
                return;
            // Initializes the TPCANTPMsg to send with the selected mapping.
            msg.IDTYPE = mapping.m_canIdType;
            msg.FORMAT = mapping.m_formatType;
            msg.MSGTYPE = mapping.m_msgType;
            msg.TA_TYPE = mapping.m_targetType;
            if (mapping.IsAutomatic)
            {
                // Be cautious with Enhanced format addressing which uses 11 bits for addresses
                if (mapping.m_formatType != TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                {
                    msg.SA = Convert.ToByte(numericUpDownSourceAddr.Value);
                    msg.TA = Convert.ToByte(numericUpDownTargetAddr.Value);
                    msg.RA = Convert.ToByte(numericUpDownRemoteAddr.Value);
                }
                else
                {
                    // Enhanced format addressing uses 11 bits for its addresses,
                    //  in this case Remote address is used to store the extra bits
                    //  from Source and Target addresses.
                    uint iBuf;
                    // Unique SA is 11bits: store bits [7..0] in SA
                    // and the rest in [7..4] of RA
                    iBuf = Convert.ToUInt32(numericUpDownSourceAddr.Value);
                    msg.SA = (byte)iBuf;
                    msg.RA = (byte)((iBuf >> 8) << 4);
                    // Unique TA is 11bits: store bits [7..0] in TA
                    // and the rest in [3..0] of RA
                    iBuf = Convert.ToUInt32(numericUpDownTargetAddr.Value);
                    msg.TA = (byte)iBuf;
                    msg.RA |= (byte)(iBuf >> 8);
                }
            }
            else
            {
                // The mapping defines the addresses.
                msg.SA = mapping.m_sourceAddr;
                msg.TA = mapping.m_targetAddr;
                msg.RA = mapping.m_remoteAddr;
            }
            // Initializes RESULT now to avoid a strange behaviour within Visual Studio while debugging (C++.NET issue).
            msg.RESULT = TPCANTPConfirmation.PCANTP_N_OK;
            // Sets length and data.
            msg.LEN = Convert.ToUInt16(numericUpDownMsgLength.Value);
            msg.DATA = new byte[4095];
            for (int i = 0; i < msg.LEN; i++)
            {
                if (i * 2 < textBoxMsgData.TextLength - 1)
                    strByte = textBoxMsgData.Text.Substring(i * 2, 2);
                else if (i * 2 < textBoxMsgData.TextLength)
                    strByte = textBoxMsgData.Text.Substring(i * 2, 1);
                else
                    strByte = "0";
                msg.DATA[i] = Convert.ToByte(strByte, 16);
            }
            // CAN FD support
            if (checkBoxFDMessage.Checked)
            {
                msg.IDTYPE |= TPCANTPIdType.PCANTP_ID_CAN_FD;
                // if BRS support
                if (checkBoxBRS.Checked)
                {
                    msg.IDTYPE |= TPCANTPIdType.PCANTP_ID_CAN_BRS;
                }
            }
            // Manage priority (available only in 29 bits mixed, enhanced or fixed normal)
            if (checkBoxHasPriority.Checked)
            {
                if ((mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL) || (mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_MIXED) ||
                     mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                {
                    if (TPCANTPIdTypePriority.PCANTP_ID_CAN_IS_EXTENDED((Byte)msg.IDTYPE))
                    {
                        msg.IDTYPE |= (TPCANTPIdType)CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK;
                        // Clean old priority
                        msg.IDTYPE &= (TPCANTPIdType)CanTpApi.PCANTP_ID_CAN_MASK + CanTpApi.PCANTP_ID_CAN_IS_PRIORITY_MASK;
                        // Add priority value
                        msg.IDTYPE |= (TPCANTPIdType)((int)numericUpDownPriority.Value << 5);
                    }
                }
            }
            // Sends the message to the Tx queue of the API.
            sts = CanTpApi.Write(m_pctpHandle, ref msg);
            CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, msg);
        }

        /// <summary>
        /// Event handler called when selection of comboBox "Write mapping" is changed.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void comboBoxMsgMapping_SelectedIndexChanged(object sender, EventArgs e)
        {
            MappingStatus mapping;
            bool enabled;

            // Asserts a mapping is selected.
            mapping = null;
            if (comboBoxMsgMapping.SelectedItem != null)
            {
                // Gets the selected mapping.
                mapping = (MappingStatus)(((ComboBoxItem)comboBoxMsgMapping.SelectedItem).Data);
                // Enables the button to write the message.
                buttonMsgWrite.Enabled = true;
                // Source (SA), Target (TA) and Remote address (RA) textboxes are 
                //  enabled ONLY if the mapping allows dynamic 
                //  generation of CAN ID based on SA, TA, RA.
                //  i.e. only if an "automatic mapping" is selected.
                enabled = mapping.IsAutomatic;
                numericUpDownSourceAddr.Enabled = enabled;
                numericUpDownTargetAddr.Enabled = enabled;
                // RA is also only available with Remote Diagnostic Messages.
                if (mapping.m_msgType == TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC)
                {
                    numericUpDownRemoteAddr.Enabled = enabled;
                }
                else
                {
                    numericUpDownRemoteAddr.Enabled = false;
                    // To avoid confusion, RA text is reset to 0x00.
                    numericUpDownRemoteAddr.Text = "0";
                }
                // Sets SA/TA maximum values (default is 8bits, but Enhanced addressing uses 11bits for SA and TA
                numericUpDownSourceAddr.Maximum = (mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED) ? 0x7FF : 0xFF;
                numericUpDownTargetAddr.Maximum = numericUpDownSourceAddr.Maximum;
                // Sets SA/TA/RA as defined in the mapping.
                // If an "automatic mapping" is selected,
                //  then the user has to set those value.
                if (!mapping.IsAutomatic)
                {
                    numericUpDownSourceAddr.Value = mapping.m_sourceAddr;
                    numericUpDownTargetAddr.Value = mapping.m_targetAddr;
                    numericUpDownRemoteAddr.Value = mapping.m_remoteAddr;
                }
                // Sets the maximum size of the ISO-TP message.
                if (mapping.m_targetType == TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL)
                {
                    // Functional addressing allows only Single Frame to be sent
                    numericUpDownMsgLength.Maximum = CanTpUtils.ISOTP_MSG_FUNC_MAX_LENGTH;
                    if (mapping.m_formatType != TPCANTPFormatType.PCANTP_FORMAT_NORMAL &&
                        mapping.m_formatType != TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL)
                        numericUpDownMsgLength.Maximum--;
                }
                else
                    numericUpDownMsgLength.Maximum = CanTpUtils.ISOTP_MSG_PHYS_MAX_LENGTH;
                if (((mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL) || (mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_MIXED) ||
                      mapping.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED) && mapping.m_canIdType == TPCANTPIdType.PCANTP_ID_CAN_29BIT)
                {
                    // If it is a 29bits mapping fixed normal, enhanced or mixed, changing "Priority" value is allowed
                    checkBoxHasPriority.Enabled = true;
                }
                else
                {
                    checkBoxHasPriority.Enabled = false;
                    checkBoxHasPriority.Checked = false;
                    numericUpDownPriority.Enabled = false;
               }
            }
            // No mapping is selected in comboBox.
            else
            {
                // Disables "write message" UI components.
                enabled = false;
                buttonMsgWrite.Enabled = enabled;
                numericUpDownSourceAddr.Enabled = enabled;
                numericUpDownTargetAddr.Enabled = enabled;
                numericUpDownRemoteAddr.Enabled = enabled;
            }
        }

        /// <summary>
        /// Event handler called when numericUpDown of "Message's length" is changed,
        /// or textBoxMsgData is left.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void numericUpDownMsgLength_ValueChanged(object sender, EventArgs e)
        {
            // Gets the size of the text based on the data length.
            // Note: each byte is translated in 2 characters.
            textBoxMsgData.MaxLength = Convert.ToInt32(numericUpDownMsgLength.Value) * 2;
            // Adds a zero to the last "non-two-character" byte (ex. 0xF => 0x0F ; 0xFA1 => 0xFA01)
            if (textBoxMsgData.Text.Length % 2 == 1)
                textBoxMsgData.Text = textBoxMsgData.Text.Insert(textBoxMsgData.Text.Length - 1, "0");
            // Modify data string based on the new length
            if (textBoxMsgData.Text.Length > textBoxMsgData.MaxLength)
                // Data is longer than the length, truncates the string
                textBoxMsgData.Text = textBoxMsgData.Text.Substring(0, textBoxMsgData.MaxLength);
            else
                // Data is shroter than the length, pads the string with zeros
                textBoxMsgData.Text = textBoxMsgData.Text.PadRight(textBoxMsgData.MaxLength, '0');
        }

        /// <summary>
        /// Event handler called when a key is pressed inside "Write message Data" textBox.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">Information on the event.</param>
        private void textBoxHexOnly_KeyPress(object sender, KeyPressEventArgs e)
        {
            char chCheck;

            // Converts the Character to its Upper case equivalent
            chCheck = char.ToUpper(e.KeyChar);
            // Key is the Delete (Backspace) Key
            if (chCheck == 8)
                return;
            e.KeyChar = chCheck;
            // Key is a number between 0-9
            if ((chCheck > 47) && (chCheck < 58))
                return;
            // Key is a character between A-F
            if ((chCheck > 64) && (chCheck < 71))
                return;
            // key is neither a number nor a character between A(a) and F(f)
            e.Handled = true;
        }
        #endregion

        #region Controllers handling ISO-TP connection
        /// <summary>
        /// Connects to a PCAN ISO-TP channel and sets m_pctpHandle.
        /// </summary>
        /// <param name="canHandle">A PCANTP Channel Handle representing a PCANTP-Client</param>
        /// <param name="baudrate">The CAN Hardware speed</param>
        /// <param name="hwType">Non plug-'n-play: The type of hardware and operation mode</param>
        /// <param name="ioPort">Non plug-'n-play: The I/O address for the parallel port</param>
        /// <param name="interrupt">Non plug-'n-play: Interrupt number of the parallel port</param>
        /// <returns>true is returned on success</returns>
        private bool connect(TPCANTPHandle canHandle, TPCANTPBaudrate baudrate, TPCANTPHWType hwType, uint ioPort, byte interrupt)
        {
            TPCANTPStatus sts;

            // Connects to the selected PCAN-ISO-TP channel.
            // If FD support
            if (checkBoxCanFd.Checked)
            {
                // Convert unicode string to multibytes char
                sts = CanTpApi.InitializeFD(canHandle, (System.String)textBoxCanFdBitrate.Text);
                CanTpUtils.checkCanTpStatus(canHandle, sts, Encoding.ASCII.GetBytes(textBoxCanFdBitrate.Text.ToString()));
            }
            else
            {
                sts = CanTpApi.Initialize(canHandle, baudrate, hwType, ioPort, interrupt);
                CanTpUtils.checkCanTpStatus(canHandle, sts, (int)baudrate, (int)hwType, (int)ioPort, interrupt);
            }

            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
            {
                // Stores the currently connected channel.
                m_pctpHandle = canHandle;
                // Clear mappings.
                mappingsInit();
                mappingsChanged();
                // Desable FD checkbox
                checkBoxCanFd.Enabled = false;
            }
            else
                m_pctpHandle = CanTpApi.PCANTP_NONEBUS;
            // Sets the connection status of the main-form.
            setConnectionStatus(sts == TPCANTPStatus.PCANTP_ERROR_OK);

            //CCANExamplesValidation.displayValidatioMsg(sts, canHandle, baudrate, hwType, ioPort, interrupt, CCANExamplesValidation.CallerName());
            return (m_pctpHandle != CanTpApi.PCANTP_NONEBUS);
        }

        private bool readMappingFile()
        {
            MappingStatus mapping;
            try
            {
                if (System.IO.File.Exists(@"Mappings.csv") == true)
                {
                    // Open the file to read from.
                    string[] readText = System.IO.File.ReadAllLines(@"Mappings.csv");
                    bool skipFirstLine = true; // First line is CSV header
                    foreach (string s in readText)
                    {
                        if (skipFirstLine == true)
                        {
                            skipFirstLine = false;
                            continue;
                        }
                        String[] allValues = s.Split(';');
                        if (allValues.Length == 9) // Mapping must have 9 elements to be valid
                        {
                            //            0      1                  2           3    4  5   6       7     8
                            //newLine = "CAN_ID;CAN_ID response;Target Type;ID Type;SA;TA;MsgType;Format;RA" + Environment.NewLine;
                            //mapping = new MappingStatus(0, 1, 3, 7, 6, 2, 4, 5, 8)
                            mapping = new MappingStatus(UInt32.Parse(allValues[0]),
                                                        UInt32.Parse(allValues[1]),
                                                        (TPCANTPIdType)UInt32.Parse(allValues[3]),
                                                        (TPCANTPFormatType)UInt32.Parse(allValues[7]),
                                                        (TPCANTPMessageType)UInt32.Parse(allValues[6]),
                                                        (TPCANTPAddressingType)UInt32.Parse(allValues[2]),
                                                        Byte.Parse(allValues[4]),
                                                        Byte.Parse(allValues[5]),
                                                        Byte.Parse(allValues[8]));
                            mappingsAdd(mapping);
                            mappingsChanged();
                        }
                        //m_mappings.Add(mapping);
                    }
                }
            }
            catch { };
            return true;
        }

        /// <summary>
        /// Disconnects the currently connected channel (m_pctpHandle)
        /// </summary>
        /// <returns>true is returned on success</returns>
        private bool disconnect()
        {
            TPCANTPStatus sts;

            // Disconnects from selected PCAN-ISO-TP channel
            sts = CanTpApi.Uninitialize(m_pctpHandle);
            CanTpUtils.checkCanTpStatus(m_pctpHandle, sts);
            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
            {
                // Resets channel and stored mappings.
                m_pctpHandle = CanTpApi.PCANTP_NONEBUS;
                // Resets stored mappings.
                m_mappings.Clear();
                mappingsChanged();
                checkBoxHasPriority.Checked = false;
                checkBoxHasPriority.Enabled = false;
                numericUpDownPriority.Enabled = false;
                // enable FD checkbox
                checkBoxCanFd.Enabled = true;
            }
            // Sets the connection status of the main-form
            setConnectionStatus(sts != TPCANTPStatus.PCANTP_ERROR_OK);
            //CCANExamplesValidation.displayValidatioMsg(sts, m_pctpHandle, 0, 0, 0, 0, CCANExamplesValidation.CallerName());
            return isConnected();
        }
        #endregion

        #region Controllers handling ISO-TP mappings
        /// <summary>
        /// Sets the default "automatic" mappings.
        /// Those mappings are set for tutorial/understanding purpose,
        /// ISO-TP API handles internally CANTP messages with those
        /// network information and do not require mappings to be defined.
        /// </summary>
        private void mappingsInit()
        {
            MappingStatus mapping;

            // Initializes the internal list of mappings
            if (m_mappings == null)
                m_mappings = new List<MappingStatus>();
            else
                m_mappings.Clear();

            // 29 bits CAN ID, Normal Fixed and physical addressing
            mapping = new MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
                TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0);
            m_mappings.Add(mapping);
            // 29 bits CAN ID, Normal Fixed and functional addressing
            mapping = new MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
                TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL, 0, 0, 0);
            m_mappings.Add(mapping);
            // 29 bits CAN ID, Mixed and physical addressing
            mapping = new MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
                TPCANTPFormatType.PCANTP_FORMAT_MIXED, TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0);
            m_mappings.Add(mapping);
            // 29 bits CAN ID, Mixed and functional addressing
            mapping = new MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
                TPCANTPFormatType.PCANTP_FORMAT_MIXED, TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL, 0, 0, 0);
            m_mappings.Add(mapping);
            mapping = new MappingStatus(0, 0, TPCANTPIdType.PCANTP_ID_CAN_29BIT,
            // 29 bits CAN ID, Enhanced and physical addressing (ISO 15765-3)
                TPCANTPFormatType.PCANTP_FORMAT_ENHANCED, TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC,
                TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL, 0, 0, 0);
            m_mappings.Add(mapping);
        }

        /// <summary>
        /// Sets a new mapping to the ISO-TP API.
        /// </summary>
        /// <param name="mapping"></param>
        private void mappingsAdd(MappingStatus mapping)
        {
            TPCANTPStatus sts;

            // Adds the mapping inside the API
            sts = CanTpApi.AddMapping(m_pctpHandle, mapping.m_canId, mapping.m_canIdResponse,
                   mapping.m_canIdType, mapping.m_formatType, mapping.m_msgType,
                   mapping.m_sourceAddr, mapping.m_targetAddr, mapping.m_targetType, mapping.m_remoteAddr);

            CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, mapping);
            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
            {
                // Adds the mapping to the internal list of configured mappings.
                m_mappings.Add(mapping);
            }

        }


        /// <summary>
        /// Manage FD checkboxes when FD support is checked
        /// </summary>
        public void ManageUIForFD(bool p_state)
        {
            checkBoxBRS.Visible = p_state;
            checkBoxFDMessage.Visible = p_state;
            checkBoxBRS.Enabled = !p_state;
            checkBoxFDMessage.Enabled = p_state;
        }

        /// <summary>
        /// Function to be called when a change occured in the configured mappings.
        /// The function updates all UI components according to the new settings.
        /// </summary>
        private void mappingsChanged()
        {
            MappingStatus mappingComboSel;   // selected mapping in comboBoxMsgMapping
            MappingStatus mappingListSel;    // selected mapping in listViewMappings
            MappingStatus mapping;
            ListViewItem lvi;

            mappingComboSel = null;
            mappingListSel = null;
            // Store selected item in mappingListSel.
            if (listViewMappings.SelectedItems.Count > 0)
                mappingListSel = (MappingStatus)(listViewMappings.SelectedItems[0].Tag);
            // Store selected item in comboBoxMsgMapping.
            if (comboBoxMsgMapping.SelectedItem != null)
                mappingComboSel = (MappingStatus)((ComboBoxItem)comboBoxMsgMapping.SelectedItem).Data;

            // Clear listview of mappings.
            listViewMappings.Items.Clear();
            // Clear comboBox of mappings in "write message" section.
            comboBoxMsgMapping.Items.Clear();
            // Loop through all configured mappings and add them to UI.
            for (int i = 0; i < m_mappings.Count; i++)
            {
                mapping = m_mappings[i];
                // Add mapping to listView of mappings.
                lvi = new ListViewItem();
                lvi.Tag = mapping;
                lvi.Text = mapping.CanId;
                lvi.SubItems.Add(mapping.CanIdResponse);
                lvi.SubItems.Add(mapping.TargetType);
                lvi.SubItems.Add(mapping.CanIdType);
                lvi.SubItems.Add(mapping.SourceAddress);
                lvi.SubItems.Add(mapping.TargetAddress);
                lvi.SubItems.Add(mapping.MsgType);
                lvi.SubItems.Add(mapping.FormatType);
                lvi.SubItems.Add(mapping.RemoteAddress);
                listViewMappings.Items.Add(lvi);
                if (mapping.Equals(mappingComboSel))
                    lvi.Selected = true;
                // Add mapping to comboBox of mappings in the messages/write tab.
                comboBoxMsgMapping.Items.Add(new ComboBoxItem(mapping.Name, mapping));
                if (mapping.Equals(mappingComboSel))
                    comboBoxMsgMapping.SelectedIndex = i;
            }
        }
        #endregion

        #region Controllers handling ISO-TP messages
        /// <summary>
        /// Reads several ISO-TP messages from the bus CAN.
        /// </summary>
        private void readMessages()
        {
            TPCANTPStatus sts;

            // We read at least one time the queue looking for messages.
            // If a message is found, we look again trying to find more.
            // If the queue is empty or an error occurr, we get out from
            // the dowhile statement.
            do
            {
                sts = readMessage();
            } while (isConnected() && sts == TPCANTPStatus.PCANTP_ERROR_OK);
        }

        /// <summary>
        /// Reads an ISO-TP messages from the bus CAN.
        /// </summary>
        /// <returns>A TPCANTPStatus error code</returns>
        private TPCANTPStatus readMessage()
        {
            TPCANTPMsg msg;
            TPCANTPTimestamp ts;
            TPCANTPStatus sts;

            // Reads and process a single ISO-TP message
            sts = CanTpApi.Read(m_pctpHandle, out msg, out ts);
            if (sts == TPCANTPStatus.PCANTP_ERROR_OK)
            {
                processMessage(msg, ts);
                CanTpUtils.checkCanTpStatus(m_pctpHandle, sts, msg);
            }
            return sts;
        }

        /// <summary>
        /// Processes a received message, in order to show it in the Message-ListView
        /// </summary>
        /// <param name="msgRcvd">The received ISO-TP message</param>
        private void processMessage(TPCANTPMsg msgRcvd, TPCANTPTimestamp ts)
        {
            // Prevent concurrent access (reading thread and UI event can alter the list)
            lock (m_receiveMsgs.SyncRoot)
            {
                // Searches if the message has already been received 
                //  (same Network Address Information) or if it is a new message.
                foreach (MessageStatus msg in m_receiveMsgs)
                {
                    if (msg.isSameNetAddrInfo(msgRcvd))
                    {
                        // Modifies the existing message and exit
                        msg.update(msgRcvd, ts);
                        return;
                    }
                }
                // Message not found, it will be created
                insertMsgEntry(msgRcvd, ts);
            }
        }

        /// <summary>
        /// Inserts a new entry for a new message in the Message-ListView
        /// </summary>
        /// <param name="msg">The messasge to be inserted</param>
        /// <param name="ts">The Timesamp of the new message</param>
        private void insertMsgEntry(TPCANTPMsg msg, TPCANTPTimestamp ts)
        {
            MessageStatus msgSts;   // MessageStatus corresponding to the new TPCANTPMsg
            ListViewItem lvi;       // ListView Item corresponding to the MessageStatus
            ListViewItem.ListViewSubItem lviSub;    // SubItem of the listView Item

            lock (m_receiveMsgs.SyncRoot)
            {
                // Adds a new MessageStatus in the received-message list.
                msgSts = new MessageStatus(msg, ts, listViewMsgs.Items.Count);
                msgSts.ShowingPeriod = checkBoxMsgShowPeriod.Checked;
                m_receiveMsgs.Add(msgSts);
                // Adds a corresponding new item in the message listView
                lvi = listViewMsgs.Items.Add(msgSts.SourceAddress);
                lvi.UseItemStyleForSubItems = false;
                lvi.SubItems.Add(msgSts.TargetAddress);
                lvi.SubItems.Add(msgSts.RemoteAddress);
                lvi.SubItems.Add(msgSts.CanIdType);
                lvi.SubItems.Add(msgSts.MsgType);
                lvi.SubItems.Add(msgSts.FormatType);
                lvi.SubItems.Add(msgSts.TargetType);
                lviSub = lvi.SubItems.Add(msgSts.ResultString);
                // Highlights network ISO-TP error in red
                if (msgSts.CanTpMsg.RESULT != TPCANTPConfirmation.PCANTP_N_OK)
                    lviSub.ForeColor = Color.Red;
                lvi.SubItems.Add(msgSts.LengthString);
                lvi.SubItems.Add(msgSts.Count.ToString());
                lvi.SubItems.Add(msgSts.TimeString);
                lvi.SubItems.Add(msgSts.DataString);
            }
        }
        /// <summary>
        /// Updates the "received-message" listView with the internal list of received message.
        /// </summary>
        private void displayMessages()
        {
            ListViewItem lvi;   // listView item corresponding to a specific messageStatus

            lock (m_receiveMsgs.SyncRoot)
            {
                // Loops through all the received messageStatus to update the display
                foreach (MessageStatus msgStatus in m_receiveMsgs)
                {
                    // Checks that the data needs to be actualized
                    if (msgStatus.MarkedAsUpdated)
                    {
                        msgStatus.MarkedAsUpdated = false;
                        // Retrieves the listView corresponding to the messageStatus
                        lvi = listViewMsgs.Items[msgStatus.Position];
                        // Updates the listView item's fields
                        lvi.SubItems[0].Text = msgStatus.SourceAddress;
                        lvi.SubItems[1].Text = msgStatus.TargetAddress;
                        lvi.SubItems[2].Text = msgStatus.RemoteAddress;
                        lvi.SubItems[3].Text = msgStatus.CanIdType;
                        lvi.SubItems[4].Text = msgStatus.MsgType;
                        lvi.SubItems[5].Text = msgStatus.FormatType;
                        lvi.SubItems[6].Text = msgStatus.TargetType;
                        lvi.SubItems[7].Text = msgStatus.ResultString;
                        // Updates the Network Result color (red for an error)
                        if (msgStatus.CanTpMsg.RESULT != TPCANTPConfirmation.PCANTP_N_OK)
                            lvi.SubItems[7].ForeColor = Color.Red;
                        else
                            lvi.SubItems[7].ForeColor = lvi.ForeColor;
                        lvi.SubItems[8].Text = msgStatus.LengthString;
                        lvi.SubItems[9].Text = msgStatus.Count.ToString();
                        lvi.SubItems[10].Text = msgStatus.TimeString;
                        lvi.SubItems[11].Text = msgStatus.DataString;
                    }
                }
            }
        }
        #endregion

        #region Controllers handling thread to read message
        /// <summary>
        /// Main entry point for thread "read messages"
        /// </summary>
        private void pctpReadThread()
        {
            UInt32 iBuffer;
            TPCANTPStatus sts;

            // Sets the handle of the Receive-Event.
            iBuffer = Convert.ToUInt32(m_receiveEvent.SafeWaitHandle.DangerousGetHandle().ToInt32());
            sts = CanTpApi.SetValue(m_pctpHandle, TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT, ref iBuffer, sizeof(UInt32));
            if (sts != TPCANTPStatus.PCANTP_ERROR_OK)
            {
                // Handles failure on the UI thread then aborts thread.
                this.Invoke(m_readFailedDelegate, sts);
                return;
            }
            // Loops while the "read thread" mode is selected
            while (radioButtonMsgEvent.Checked)
            {
                // Waits for Receive-Event
                if (m_receiveEvent.WaitOne(50))
                    // Process Receive-Event using .NET Invoke function
                    // in order to interact with Winforms UI (calling the 
                    // function readMessages)
                    this.Invoke(m_readDelegate);
            }
        }
        /// <summary>
        /// Handles failure while reading messages with a thread.
        /// </summary>
        /// <param name="sts"></param>
        private void readThreadFailed(TPCANTPStatus sts)
        {
            // Shows the error status.
            CanTpUtils.checkCanTpStatus(sts);
            // reverts to the default reading method.
            radioButtonMsgTimer.PerformClick();
        }
        #endregion

        #region Controllers handling miscellaneous UI updates.
        /// <summary>
        /// Includes a new line of text into the information ListBox
        /// </summary>
        /// <param name="strMsg">Text to be included</param>
        private void includeTextMessage(string strMsg)
        {
            listBoxParamInfo.Items.Add(strMsg);
            listBoxParamInfo.SelectedIndex = listBoxParamInfo.Items.Count - 1;
        }

        /// <summary>
        /// First initialization of UI components of the "Connection" tab.
        /// </summary>
        private void initializeUiConnection()
        {
            // fill combobox "comboBoxBaudrate"
            comboBoxBaudrate.Items.Clear();
            foreach (TPCANTPBaudrate value in Enum.GetValues(typeof(TPCANTPBaudrate)))
                comboBoxBaudrate.Items.Add(new ComboBoxItem(CanTpUtils.GetBitrate(value), value));
            comboBoxBaudrate.SelectedIndex = 2;
            // fill combobox "comboBoxHwType"
            comboBoxHwType.Items.Clear();
            foreach (TPCANTPHWType value in Enum.GetValues(typeof(TPCANTPHWType)))
                comboBoxHwType.Items.Add(new ComboBoxItem(Enum.GetName(typeof(TPCANTPHWType), value), value));
            // fill combobox "comboBoxIoPort"
            comboBoxIoPort.Items.Clear();
            comboBoxIoPort.Items.Add(new ComboBoxItem("0100", 0x0100));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0120", 0x0120));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0140", 0x0140));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0200", 0x0200));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0220", 0x0220));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0240", 0x0240));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0260", 0x0260));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0278", 0x0278));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0280", 0x0280));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02A0", 0x02A0));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02C0", 0x02C0));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02E0", 0x02E0));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02E8", 0x02E8));
            comboBoxIoPort.Items.Add(new ComboBoxItem("02F8", 0x02F8));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0300", 0x0300));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0320", 0x0320));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0340", 0x0340));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0360", 0x0360));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0378", 0x0378));
            comboBoxIoPort.Items.Add(new ComboBoxItem("0380", 0x0380));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03BC", 0x03BC));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03E0", 0x03E0));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03E8", 0x03E8));
            comboBoxIoPort.Items.Add(new ComboBoxItem("03F8", 0x03F8));
            // fill combobox "comboBoxInterrupt"
            comboBoxInterrupt.Items.Clear();
            comboBoxInterrupt.Items.Add(new ComboBoxItem("3", 3));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("4", 4));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("5", 5));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("7", 7));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("9", 9));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("10", 10));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("11", 11));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("12", 12));
            comboBoxInterrupt.Items.Add(new ComboBoxItem("15", 15));
            // fill combobox "comboBoxChannel"
            buttonHwRefresh_Click(this, EventArgs.Empty);
            // select last item
            comboBoxChannel.SelectedIndex = comboBoxChannel.Items.Count - 1;
        }
        /// <summary>
        /// First initialization of UI components of the "Message" tab.
        /// </summary>
        private void initializeUiTabMessages()
        {
            numericUpDownMsgLength_ValueChanged(this, EventArgs.Empty);
        }
        /// <summary>
        /// First initialization of UI components of the "Parameter" tab.
        /// </summary>
        private void initializeUiTabParameters()
        {
            TPCANTPParameter param;
            // fill combobox "comboBoxParameter"
            comboBoxParameter.Items.Clear();
            param = TPCANTPParameter.PCANTP_PARAM_API_VERSION;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_DEBUG;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));

            param = TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_MSG_PENDING;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));

            param = TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_WFT_MAX;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));

            param = TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            param = TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            //J1939 Priority
            param = TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
            // DLC when FD is selected
            param = TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL;
            comboBoxParameter.Items.Add(new ComboBoxItem(CanTpUtils.GetParameter(param), param));
        }

        /// <summary>
        /// Activates/deactivates the different controls of the main-form according
        /// with the current connection status
        /// </summary>
        /// <param name="bConnected">Current status. True if connected, false otherwise</param>
        private void setConnectionStatus(bool bConnected)
        {
            // Connection components
            buttonInit.Enabled = !bConnected;
            buttonRelease.Enabled = bConnected;
            buttonHwRefresh.Enabled = !bConnected;
            comboBoxChannel.Enabled = !bConnected;
            comboBoxBaudrate.Enabled = !bConnected;
            comboBoxHwType.Enabled = !bConnected;
            comboBoxIoPort.Enabled = !bConnected;
            comboBoxInterrupt.Enabled = !bConnected;
            // Parameters components
            buttonParamReset.Enabled = bConnected;
            buttonParamStatus.Enabled = bConnected;
            comboBoxParameter_SelectedIndexChanged(this, EventArgs.Empty);
            // Mappings components
            buttonMappingAdd.Enabled = bConnected;
            buttonMappingDel.Enabled = bConnected;
            buttonMappingSample.Enabled = bConnected;
            listViewMappings.Enabled = bConnected;
            listViewMappings_SelectedIndexChanged(this, EventArgs.Empty);
            // Messages components
            buttonMsgRead.Enabled = bConnected;
            buttonMsgWrite.Enabled = bConnected;
            comboBoxMsgMapping.Enabled = bConnected;
            listViewMsgs.Enabled = bConnected;
            comboBoxMsgMapping_SelectedIndexChanged(this, EventArgs.Empty);

            if (!bConnected)
                // reload hardware configuration if NOT connected
                comboBoxChannel_SelectedIndexChanged(this, EventArgs.Empty);
            else
                // update reading method if connected
                radioButtonMsgRead_CheckedChanged(this, EventArgs.Empty);
        }

        /// <summary>
        /// States if an ISO-TP channel is currently connected.
        /// </summary>
        /// <returns>true if a channel is connected.</returns>
        private bool isConnected() { return m_pctpHandle != CanTpApi.PCANTP_NONEBUS; }
        #endregion

        private void buttonMappingSave_Click(object sender, EventArgs e)
        {
            try
            {
                string CSVHeader = "CAN_ID;CAN_ID response;Target Type;ID Type;SA;TA;MsgType;Format;RA" + Environment.NewLine; //"";
                string newLine = "";
                if (System.IO.File.Exists(@"Mappings.csv") == false)
                {
                    //CSVHeader = "CAN_ID;CAN_ID response;Target Type;ID Type;SA;TA;MsgType;Format;RA"; // + Environment.NewLine;
                }
                using (System.IO.StreamWriter mappingFileWriter = new System.IO.StreamWriter(@"Mappings.csv", false))
                {
                    mappingFileWriter.WriteLine(CSVHeader);
                    for (int i = 5; i < m_mappings.Count; i++) // Skip 5 firsts defaults mappings
                    {
                        newLine = Convert.ToString(m_mappings[i].m_canId) + ";" + Convert.ToString(m_mappings[i].m_canIdResponse) + ";" + Convert.ToString((int)m_mappings[i].m_targetType) +
                               ";" + Convert.ToString((int)m_mappings[i].m_canIdType) + ";" + Convert.ToString((int)m_mappings[i].m_sourceAddr) +
                                ";" + Convert.ToString((int)m_mappings[i].m_targetAddr) + ";" + Convert.ToString((int)m_mappings[i].m_msgType) +
                              ";" + Convert.ToString((int)m_mappings[i].m_formatType) + ";" + Convert.ToString((int)m_mappings[i].m_remoteAddr);
                        mappingFileWriter.WriteLine(newLine);
                    }
                }

            }
            catch { };
        }

        private void buttonMappingLoad_Click(object sender, EventArgs e)
        {
            readMappingFile();
        }

        private void checkBoxCanFd_CheckedChanged(object sender, EventArgs e)
        {
            bool bIsChecked = checkBoxCanFd.Checked;

            labelConnBaudrate.Visible = !bIsChecked;
            labelConnIoPort.Visible = !bIsChecked;
            labelConnHwType.Visible = !bIsChecked;
            labelConnInterrupt.Visible = !bIsChecked;

            comboBoxBaudrate.Visible = !bIsChecked;
            comboBoxIoPort.Visible = !bIsChecked;
            comboBoxHwType.Visible = !bIsChecked;
            comboBoxInterrupt.Visible = !bIsChecked;

            textBoxCanFdBitrate.Visible = bIsChecked;
            labelConnBitRate.Visible = bIsChecked;

            //Manage FD, BRS and Priority checkboxes (in messages tabulation)
            checkBoxFDMessage.Visible = bIsChecked;
            checkBoxBRS.Visible = bIsChecked;
            checkBoxFDMessage.Checked = false;
            checkBoxHasPriority.Checked = false;
            checkBoxBRS.Checked = false;
            ManageUIForFD(bIsChecked);
        }

        private void checkBoxFDMessage_CheckedChanged(object sender, EventArgs e)
        {
            checkBoxBRS.Enabled = checkBoxFDMessage.Checked == true;
        }

        private void checkBoxHasPriority_CheckedChanged(object sender, EventArgs e)
        {
            numericUpDownPriority.Enabled = checkBoxHasPriority.Checked;
        }
    }


    /// <summary>
    /// Represents a PCAN device
    /// </summary>
    public enum TPCANDevice : byte
    {
        /// <summary>
        /// Undefined, unknown or not selected PCAN device value
        /// </summary>
        PCAN_NONE = 0,
        /// <summary>
        /// PCAN Non-Plug and Play devices. NOT USED WITHIN PCAN-Basic API
        /// </summary>
        PCAN_PEAKCAN = 1,
        /// <summary>
        /// PCAN-ISA, PCAN-PC/104, and PCAN-PC/104-Plus
        /// </summary>
        PCAN_ISA = 2,
        /// <summary>
        /// PCAN-Dongle
        /// </summary>
        PCAN_DNG = 3,
        /// <summary>
        /// PCAN-PCI, PCAN-cPCI, PCAN-miniPCI, and PCAN-PCI Express
        /// </summary>
        PCAN_PCI = 4,
        /// <summary>
        /// PCAN-USB and PCAN-USB Pro
        /// </summary>
        PCAN_USB = 5,
        /// <summary>
        /// PCAN-PC Card
        /// </summary>
        PCAN_PCC = 6,
        /// <summary>
        /// PCAN Virtual hardware. NOT USED WITHIN PCAN-Basic API
        /// </summary>
        PCAN_VIRTUAL = 7,
        /// <summary>
        /// PCAN Gateway devices
        /// </summary>
        PCAN_LAN = 8
    }

    /// <summary>
    /// A static class providing various fonctions to format ISO-TP data.
    /// </summary>
    public static class CanTpUtils
    {
        /// <summary>
        /// Maximum data length of a physical ISO-TP message.
        /// </summary>
        public const int ISOTP_MSG_PHYS_MAX_LENGTH = 4095;
        /// <summary>
        /// Maximum data length of a functional ISO-TP message.
        /// </summary>
        public const int ISOTP_MSG_FUNC_MAX_LENGTH = 7;

#if _DEBUG_WRITE_CHECK_FILE
        /// <summary>
        /// Global variable used to know if check file initialization is done
        /// </summary>
        public static bool g_FileInitAlreadyDone = false;
        /// <summary>
        /// Global File object
        /// </summary>
        static System.IO.StreamWriter g_CheckFileToWrite;
#endif
        /// <summary>
        /// Gets the formated text from a PCAN-ISO-TP Btr0/Btr1 code.
        /// </summary>
        /// <param name="handle">PCAN-ISO-TP Baudrate to format</param>
        /// <returns>The formatted text for a baudrate</returns>
        public static string GetBitrate(TPCANTPBaudrate value)
        {
            switch (value)
            {
                case TPCANTPBaudrate.PCANTP_BAUD_1M:
                    return "1 MBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_800K:
                    return "800 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_500K:
                    return "500 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_250K:
                    return "250 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_125K:
                    return "125 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_100K:
                    return "100 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_95K:
                    return "95,238 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_83K:
                    return "83,333 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_50K:
                    return "50 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_47K:
                    return "47,619 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_33K:
                    return "33,333 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_20K:
                    return "20 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_10K:
                    return "10 kBit/s";
                case TPCANTPBaudrate.PCANTP_BAUD_5K:
                    return "5 kBit/s";
            }
            return value.ToString();
        }
        /// <summary>
        /// Gets the formated text for a PCAN-ISO-TP channel handle.
        /// </summary>
        /// <param name="handle">PCAN-ISO-TP Handle to format</param>
        /// <returns>The formatted text for a channel</returns>
        public static string GetChannelName(TPCANTPHandle handle)
        {
            TPCANDevice devDevice;
            byte byChannel;

            // Gets the owner device and channel for a 
            // PCAN-Basic handle
            if (handle < 0x100)
            {
                devDevice = (TPCANDevice)(handle >> 4);
                byChannel = (byte)(handle & 0xF);
            }
            else
            {
                devDevice = (TPCANDevice)(handle >> 8);
                byChannel = (byte)(handle & 0xFF);
            }
            // Constructs the PCAN-Basic Channel name and return it
            return string.Format("{0} {1} ({2:X2}h)", devDevice, byChannel, handle);
        }
        /// <summary>
        /// Gets the formated text from a TPCANTPParameter value.
        /// </summary>
        /// <param name="handle">Parameter to format.</param>
        /// <returns>The parameter as a text.</returns>
        public static string GetParameter(TPCANTPParameter param)
        {
            switch (param)
            {
                case TPCANTPParameter.PCANTP_PARAM_API_VERSION:
                    return "API version";
                case TPCANTPParameter.PCANTP_PARAM_BLOCK_SIZE:
                    return "ISO-TP Block Size (BS)";
                case TPCANTPParameter.PCANTP_PARAM_CAN_DATA_PADDING:
                    return "CAN Data Padding";
                case TPCANTPParameter.PCANTP_PARAM_CHANNEL_CONDITION:
                    return "Channel condition";
                case TPCANTPParameter.PCANTP_PARAM_DEBUG:
                    return "Debug mode";
                case TPCANTPParameter.PCANTP_PARAM_MSG_PENDING:
                    return "Message pending notification";
                case TPCANTPParameter.PCANTP_PARAM_PADDING_VALUE:
                    return "CAN Data Padding value";
                case TPCANTPParameter.PCANTP_PARAM_RECEIVE_EVENT:
                    return "Receive event";
                case TPCANTPParameter.PCANTP_PARAM_SEPARATION_TIME:
                    return "ISO-TP Separation time (STMin)";
                case TPCANTPParameter.PCANTP_PARAM_WFT_MAX:
                    return "ISO-TP FC.Wait frame max. (N_WFTmax)";
                case TPCANTPParameter.PCANTP_PARAM_J1939_PRIORITY:
                    return "ISO-TP J1939 priority";
                case TPCANTPParameter.PCANTP_PARAM_CAN_TX_DL:
                    return "ISO-TP FD Data Length Code (DLC)";
            }
            return "Unknown parameter";
        }
        /// <summary>
        /// Gets the formatted text of a TPCANTPStatus error.
        /// </summary>
        /// <param name="error">Error code to be translated</param>
        /// <returns>A text with the translated error</returns>
        public static string GetError(TPCANTPStatus error)
        {
            StringBuilder strTemp;

            // Creates a buffer big enough for a error-text
            strTemp = new StringBuilder(256);
            // Gets the text using the GetErrorText API function
            // If the function is successful, the translated error is returned. 
            // If it fails, a text describing the current error is returned.
            if (CanTpApi.GetErrorText(error, 0, strTemp) != TPCANTPStatus.PCANTP_ERROR_OK)
                return string.Format("An error occurred. Error-code's text ({0:X}) couldn't be retrieved", error);
            else
                return strTemp.ToString();
        }
        /// <summary>
        /// Gets the formated text of a TPCANTPFormatType value.
        /// </summary>
        /// <param name="format">value to format</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetFormatType(TPCANTPFormatType format)
        {
            switch (format)
            {
                case TPCANTPFormatType.PCANTP_FORMAT_ENHANCED:
                    return "Enhanced";
                case TPCANTPFormatType.PCANTP_FORMAT_EXTENDED:
                    return "Extended";
                case TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL:
                    return "Fixed Normal";
                case TPCANTPFormatType.PCANTP_FORMAT_MIXED:
                    return "Mixed";
                case TPCANTPFormatType.PCANTP_FORMAT_NORMAL:
                    return "Normal";
                case TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN:
                default:
                    return "Unknown";
            }
        }
        /// <summary>
        /// Gets the formated text of a timestamp.
        /// </summary>
        /// <param name="ts">Timestamp to format</param>
        /// <param name="showPeriod">States if the output is the period between 2 timestamps.</param>
        /// <param name="tsOld">If showPeriod is true, the period is based on that previous timestamp</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetTimeString(TPCANTPTimestamp ts, bool showPeriod, TPCANTPTimestamp tsOld)
        {
            double fTime;

            fTime = ts.millis + (ts.micros / 1000.0);
            if (showPeriod)
                fTime -= (tsOld.millis + (tsOld.micros / 1000.0));
            return fTime.ToString("F1");
        }
        /// <summary>
        /// Gets the data of an ISO-TP message as a formatted string.
        /// </summary>
        /// <param name="msg">ISO-TP message holding the data to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetDataString(TPCANTPMsg msg)
        {
            string strTemp;

            strTemp = "";
            for (int i = 0; i < msg.LEN; i++)
                strTemp += string.Format("{0:X2} ", msg.DATA[i]);

            return strTemp;
        }
        /// <summary>
        /// Gets a Unique Identifier based of an ISO-TP message as a formatted string.
        /// </summary>
        /// <param name="msg">The ISO-TP message.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetUniqueId(TPCANTPMsg msg)
        {
            // We format the ID of the message and show it
            return string.Format("{0:X2}h_{1:X2}h_{2}_{3}{4}_{5}_{6:X2}h",
                msg.SA, msg.TA,
                GetCanIdType(msg.IDTYPE),
                GetTargetType(msg.TA_TYPE),
                GetMsgType(msg.MSGTYPE), GetFormatType(msg.FORMAT), msg.RA);
        }
        /// <summary>
        /// Gets the CAN ID type as a formatted string.
        /// </summary>
        /// <param name="type">The can id type to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetCanIdType(TPCANTPIdType type)
        {
            string strIdTypeReturn = "";
            if ((type & TPCANTPIdType.PCANTP_ID_CAN_11BIT) == TPCANTPIdType.PCANTP_ID_CAN_11BIT)
                strIdTypeReturn = "11bits ";
            if ((type & TPCANTPIdType.PCANTP_ID_CAN_29BIT) == TPCANTPIdType.PCANTP_ID_CAN_29BIT)
                strIdTypeReturn += "29bits ";
            if ((type & TPCANTPIdType.PCANTP_ID_CAN_FD) == TPCANTPIdType.PCANTP_ID_CAN_FD)
                strIdTypeReturn += "FD ";
            if ((type & TPCANTPIdType.PCANTP_ID_CAN_BRS) == TPCANTPIdType.PCANTP_ID_CAN_BRS)
                strIdTypeReturn += "BRS ";
            if (strIdTypeReturn == "")
                strIdTypeReturn = "Unknown";
            return strIdTypeReturn;
        }
        /// <summary>
        /// Gets the ISO-TP Addressing type as a formatted string.
        /// </summary>
        /// <param name="type">The addressing type to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetTargetType(TPCANTPAddressingType type)
        {
            switch (type)
            {
                case TPCANTPAddressingType.PCANTP_ADDRESSING_FUNCTIONAL:
                    return "Functional";
                case TPCANTPAddressingType.PCANTP_ADDRESSING_PHYSICAL:
                    return "Physical";
                case TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN:
                default:
                    return "Unknown";
            }
        }
        /// <summary>
        /// Gets the ISO-TP message type as a formatted string.
        /// </summary>
        /// <param name="type">The message type to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetMsgType(TPCANTPMessageType type)
        {
            switch (type)
            {
                case TPCANTPMessageType.PCANTP_MESSAGE_DIAGNOSTIC:
                    return "Diagnostic";
                case TPCANTPMessageType.PCANTP_MESSAGE_INDICATION:
                    return "Indication";
                case TPCANTPMessageType.PCANTP_MESSAGE_INDICATION_TX:
                    return "Indication TX";
                case TPCANTPMessageType.PCANTP_MESSAGE_REMOTE_DIAGNOSTIC:
                    return "Remote Diagnostic";
                case TPCANTPMessageType.PCANTP_MESSAGE_REQUEST_CONFIRMATION:
                    return "Confirmation";
                case TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN:
                default:
                    return "Unknown";
            }
        }
        /// <summary>
        /// Gets an ISO-TP address type as a formatted string.
        /// </summary>
        /// <param name="type">The address to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetAddress(byte address)
        {
            return String.Format("{0:X2}h", address);
        }
        /// <summary>
        /// Gets an CAN ID as a formatted string.
        /// </summary>
        /// <param name="canId">The CAN ID to format.</param>
        /// <param name="isExtended">True if the CAN ID is 29bits.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetCanId(uint canId, bool isExtended)
        {
            if (!isExtended)
                return String.Format("{0:X3}h", canId);
            else
                return String.Format("{0:X8}h", canId);
        }
        /// <summary>
        /// Gets an ISO-TP Network Result as a formatted string.
        /// </summary>
        /// <param name="result">The result to format.</param>
        /// <returns>The parameter formatted as a human-readable string.</returns>
        public static string GetResult(TPCANTPConfirmation result)
        {
            string strTmp;

            switch (result)
            {
                case TPCANTPConfirmation.PCANTP_N_BUFFER_OVFLW:
                    strTmp = "Buffer overflow";
                    break;
                case TPCANTPConfirmation.PCANTP_N_ERROR:
                    strTmp = "Error";
                    break;
                case TPCANTPConfirmation.PCANTP_N_INVALID_FS:
                    strTmp = "Invalid First frame";
                    break;
                case TPCANTPConfirmation.PCANTP_N_OK:
                    strTmp = "Ok";
                    break;
                case TPCANTPConfirmation.PCANTP_N_TIMEOUT_A:
                    strTmp = "Timeout A";
                    break;
                case TPCANTPConfirmation.PCANTP_N_TIMEOUT_BS:
                    strTmp = "Timeout Bs";
                    break;
                case TPCANTPConfirmation.PCANTP_N_TIMEOUT_CR:
                    strTmp = "Timeout Cr";
                    break;
                case TPCANTPConfirmation.PCANTP_N_UNEXP_PDU:
                    strTmp = "Unexpected Protocol Data Unit";
                    break;
                case TPCANTPConfirmation.PCANTP_N_WFT_OVRN:
                    strTmp = "Wait For Transmit overrun";
                    break;
                case TPCANTPConfirmation.PCANTP_N_WRONG_SN:
                    strTmp = "Wrong Sequence Number";
                    break;
                default:
                    strTmp = "Unknown";
                    break;
            }

            return string.Format("{0} ({1})", (byte)result, strTmp);
        }

        /// <summary>
        /// this is the CCITT CRC 16 polynomial X^16  + X^12  + X^5  + 1.
        /// This works out to be 0x1021, but the way the algorithm works
        /// lets us use 0x8408 (the reverse of the bit pattern).  The high
        /// bit is always assumed to be set, thus we only use 16 bits to
        /// represent the 17 bit value.
        /// </summary>
        /// <param name="p_data">Tab of BYTE</param>
        /// <param name="p_length">Length of tab</param>
        /// <returns>CRC 16</returns>
        public static ushort crc16(byte[] p_data, ushort p_length)
        {
            byte i;
            uint indexData = 0;
            ushort data;
            ushort crc = 0xffff;
            const ushort POLY = 0x8408;

            if (p_length == 0)
                return (ushort)(~crc);

            do
            {
                data = (ushort)(0xff & p_data[indexData++]);
                for (i = 0;
                i < 8;
                i++)
                {
                    if (((crc & 0x0001) ^ (data & 0x0001)) != 0)
                        crc = (ushort)((crc >> 1) ^ POLY);
                    else crc >>= 1;
                    data >>= 1;
                }
            } while (--p_length > 0);

            crc = (ushort)~crc;
            data = crc;
            crc = (ushort)((crc << 8) | (data >> 8 & 0xff));

            return (crc);
        }

        /// <summary>
        /// Checks a TPCANTPStatus value and shows a messageBox if the status is not OK.
        /// </summary>
        /// <param name="err">ISO-TP Status to check</param>
        public static void checkCanTpStatus(TPCANTPStatus err)
        {
            if (err == TPCANTPStatus.PCANTP_ERROR_OK)
                return;
            MessageBox.Show(CanTpUtils.GetError(err) + " (" + err + ")",
                "CANTP Status", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
        }

        /// <summary>
        /// Check the status of CAN function result and give information to compare results in debug mode
        /// </summary>
        /// <param name="p_Handle">CAN Handle</param>
        /// <param name="p_Err">CAN Status</param>
        /// <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
        /// <param name="p_Msg">Pointer on CAN Message structure</param>
        public static void checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, TPCANTPMsg p_Msg, params int[] p_Args)
        {
            string strOutMsg = "";
            string strCallerName = "";
            string strData = ""; // SA[]; TA[]; TA_TYPE[]; RA[]; IDTYPE[]; MSGTYPE[]; FORMAT[]; RESULT[]; LEN[]; DATA[]";

            uint polynomialIndex = 0;
            const uint maxsize = 256;
            ushort length = (ushort)p_Args.Length;
            byte[] datatab = new byte[maxsize];
            uint result = 0;
            uint counter = 0;

#if _DEBUG_WRITE_CHECK_FILE
            string checkFileName = "";
            if (g_FileInitAlreadyDone == false)
            {
                // Init file to write result
                string sTime = DateTime.Now.ToString("yyyy.MM.dd_HH.mm.ss");
                checkFileName = "C#-" + sTime + ".csv";
                try
                {
                    g_CheckFileToWrite = new System.IO.StreamWriter(checkFileName, false);

                    //g_CheckFileToWrite.Open(checkFileName, CFile::modeCreate | CFile::modeWrite);
                    g_CheckFileToWrite.WriteLine("CallerName;Result;Args_1;Args_2;Args_3;Args_4;Args_5;Args_6;Args_7;Args_8;Args_9;Args_10;Args_11");
                    g_CheckFileToWrite.Flush();
                    g_FileInitAlreadyDone = true;
                }
                catch { };
            }
#endif

            // Get caller method name
            strCallerName = GetCallerMethodName();

            // Polynomial calculation for unique result from Args
            result += (++polynomialIndex) * p_Handle;
            result += (++polynomialIndex) * (uint)p_Err;

            strData = String.Format("Handle[{0}];Status[{1}]", p_Handle, (int)p_Err);
            for (int i = 0; i < p_Args.Length; i++)
            {
                datatab[polynomialIndex++] = (byte)(p_Args[i] * (polynomialIndex + 1));
                result += (uint)(p_Args[i] * (polynomialIndex + 1));
                strData = String.Format("{0};Arg_{1}[{2}]", strData, counter++, (int)p_Args[i]);
            }


            // If there is a message structure a CRC is calculated dans message is added in out string
            if (p_Msg.LEN > 0)
            {
                result += (++polynomialIndex) * p_Msg.SA;
                result += (++polynomialIndex) * p_Msg.TA;
                result += (++polynomialIndex) * (uint)p_Msg.TA_TYPE;
                result += (++polynomialIndex) * p_Msg.RA;
                result += (++polynomialIndex) * (uint)p_Msg.IDTYPE;
                result += (++polynomialIndex) * (uint)p_Msg.MSGTYPE;
                result += (++polynomialIndex) * (uint)p_Msg.FORMAT;
                result += (++polynomialIndex) * (uint)p_Msg.RESULT;
                result += (++polynomialIndex) * p_Msg.LEN;

                // All Data fields are used in case of bad data indexing during message construction
                //for (int i = 0; i < 4095; i++) retVal += (++p_polynomeIndex) * p_canMsg.DATA[i];
                // A CRC 16 is calculated on DATA to prevent LONG64 overflow with polynome on 4095 fields
                // And not a CRC32 for perfomances improvement
                result += crc16(p_Msg.DATA, p_Msg.LEN);
                strData = String.Format("{0};SA[{1}];TA[{2}];TA_TYPE[{3}];RA[{4}];IDTYPE[{5}];MSGTYPE[{6}];FORMAT[{7}];RESULT[{8}];LEN[{9}];DATA[",
                    strData, p_Msg.SA, p_Msg.TA, (int)p_Msg.TA_TYPE, p_Msg.RA, (int)p_Msg.IDTYPE, (int)p_Msg.MSGTYPE, (int)p_Msg.FORMAT, (int)p_Msg.RESULT, p_Msg.LEN);
                for (uint i = 0; i < p_Msg.LEN; i++)
                {
                    strData = String.Format("{0}{1}", strData, p_Msg.DATA[i].ToString("X2"));
                }
                strData = String.Format("{0}]", strData);
            }

            // Concat all strings
            strOutMsg = String.Format("{0};{1};{2}", strCallerName, result, strData);
#if _DEBUG_WRITE_CHECK_FILE
#if _DEBUG
            Console.WriteLine(strOutMsg);
#endif
            if (g_FileInitAlreadyDone == true)
            {
                // Write result in file
                try
                {
                    g_CheckFileToWrite.WriteLine(strOutMsg);
                    g_CheckFileToWrite.Flush();
                }
                catch { };
            }
#endif

            if (p_Err == TPCANTPStatus.PCANTP_ERROR_OK)
                return;
            MessageBox.Show(CanTpUtils.GetError(p_Err) + " (" + p_Err + ")",
                "CANTP Status", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
        }

        /// <summary>
        /// Check the status of CAN function result and give information to compare results in debug mode
        /// </summary>
        /// <param name="p_Handle">CAN Handle</param>
        /// <param name="p_Err">CAN Status</param>
        /// <param name="p_Args">List of arguments given to CANTP function (usage : {(int)var1, (int) var2, ... } </param>
        /// <param name="p_Msg">Pointer on CAN Message structure</param>
        public static void checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, params int[] p_Args)
        {
            TPCANTPMsg p_Msg;
            p_Msg.RA = 0;
            p_Msg.SA = 0;
            p_Msg.TA = 0;
            p_Msg.LEN = 0;
            p_Msg.DATA = new byte[4095];
            p_Msg.FORMAT = TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN;
            p_Msg.IDTYPE = TPCANTPIdType.PCANTP_ID_CAN_11BIT;
            p_Msg.MSGTYPE = TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN;
            p_Msg.RESULT = TPCANTPConfirmation.PCANTP_N_ERROR;
            p_Msg.TA_TYPE = TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN;

            checkCanTpStatus(p_Handle, p_Err, p_Msg, p_Args);
        }

        /// <summary>
        /// Check the status of CAN function result and give information to compare results in debug mode
        /// </summary>
        /// <param name="p_Handle">CAN Handle</param>
        /// <param name="p_Err">CAN Status</param>
        /// <param name="p_Msg">Characters buffer</param>
        /// <param name="p_Parameter">Parameter</param>
        public static void checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, byte[] p_Msg, byte p_Parameter = 0)
        {
            if (p_Msg.Length != 0)
            {
                CanTpUtils.checkCanTpStatus(p_Handle, p_Err, p_Parameter, crc16(p_Msg, (ushort)(p_Msg.Length)));
            }
            else
            {
                CanTpUtils.checkCanTpStatus(p_Handle, p_Err);
            }
        }

        /// <summary>
        /// Check the status of CAN function result and give information to compare results in debug mode
        /// </summary>
        /// <param name="p_Handle">CAN Handle</param>
        /// <param name="p_Err">CAN Status</param>
        /// <param name="p_Msg">Characters buffer</param>
        /// <param name="p_Parameter">Parameter</param>
        public static void checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, StringBuilder p_Msg, TPCANTPParameter p_Parameter = 0)
        {
            if (p_Msg.Length != 0)
            {
                CanTpUtils.checkCanTpStatus(p_Handle, p_Err, (int)p_Parameter, crc16(Encoding.ASCII.GetBytes(p_Msg.ToString()), (ushort)(p_Msg.Length)));
            }
            else
            {
                CanTpUtils.checkCanTpStatus(p_Handle, p_Err);
            }
        }

        /// <summary>
        /// Check the status of CAN function result and give information to compare results in debug mode
        /// </summary>
        /// <param name="p_Handle">CAN Handle</param>
        /// <param name="p_Err">CAN Status</param>
        /// <param name="p_Mapping">Pointer on mapping structure</param>
        public static void checkCanTpStatus(TPCANTPHandle p_Handle, TPCANTPStatus p_Err, MappingStatus p_Mapping)
        {
            CanTpUtils.checkCanTpStatus(p_Handle, p_Err,
                         (int)p_Mapping.m_canId, (int)p_Mapping.m_canIdResponse,
                         (int)p_Mapping.m_canIdType, (int)p_Mapping.m_formatType, (int)p_Mapping.m_msgType,
                         p_Mapping.m_sourceAddr, p_Mapping.m_targetAddr, (int)p_Mapping.m_targetType, p_Mapping.m_remoteAddr);
        }

        /// <summary>
        /// Get caller name
        /// </summary>
        /// <returns>Return the name of the caller</returns>
        public static string GetCallerMethodName()
        {
            System.Diagnostics.StackFrame frame = new System.Diagnostics.StackFrame((int)0);
            for (uint i = 0; i < 5; i++)
            {
                frame = new System.Diagnostics.StackFrame((int)i);
                if ((frame.GetMethod().Name != "checkCanTpStatus") && (frame.GetMethod().Name != "GetCallerMethodName"))
                {
                    return frame.GetMethod().Name;
                }
            }
            return "Unknown";
        }

    }

    /// <summary>
    /// Represents an item of a ComboBox
    /// </summary>
    class ComboBoxItem
    {
        /// <summary>
        /// Real data object to add to the comboBox
        /// </summary>
        public Object Data { get; set; }
        /// <summary>
        /// Text to display in the comboBox
        /// </summary>
        public String Text { get; set; }

        /// <summary>
        /// Default constructor
        /// </summary>
        /// <param name="text">Text to display.</param>
        /// <param name="data">Real data object.</param>
        public ComboBoxItem(String text, Object data)
        {
            this.Text = text;
            this.Data = data;
        }

        public override string ToString()
        {
            return Text;
        }
    }

    /// <summary>
    /// Class that stores information of a single ISO-TP mapping.
    /// </summary>
    public class MappingStatus
    {
        /// <summary>
        /// CAN ID to map to CAN-ISO-TP network addressing information
        /// </summary>
        public uint m_canId { get; set; }
        /// <summary>
        /// CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
        /// (i.e. the Flow Control frames will use this ID)
        /// </summary>
        public uint m_canIdResponse { get; set; }
        /// <summary>
        /// The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)
        /// </summary>
        public TPCANTPIdType m_canIdType { get; set; }
        /// <summary>
        /// The ISO-TP network addressing format.
        /// </summary>
        public TPCANTPFormatType m_formatType { get; set; }
        /// <summary>
        /// Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)
        /// </summary>
        public TPCANTPMessageType m_msgType { get; set; }
        /// <summary>
        /// Source address
        /// </summary>
        public byte m_sourceAddr { get; set; }
        /// <summary>
        /// Target address
        /// </summary>
        public byte m_targetAddr { get; set; }
        /// <summary>
        /// Type of addressing (physical: -node to node-, or functional: -node to all-)
        /// </summary>
        public TPCANTPAddressingType m_targetType { get; set; }
        /// <summary>
        /// Remote address (used only with remote disgnostic message)
        /// </summary>
        public byte m_remoteAddr { get; set; }

        /// <summary>
        /// Default constructor
        /// </summary>
        /// <param name="canId">CAN ID to map to CAN-ISO-TP network addressing information</param>
        /// <param name="canIdResponse">CAN ID used by the other side to internally respond to the CAN-ISO-TP segmented frames
        /// (i.e. the Flow Control frames will use this ID)</param>
        /// <param name="canIdType">The CAN ID type used by the mapping (11 bits or 29 bits CAN ID)</param>
        /// <param name="formatType">The ISO-TP network addressing format.</param>
        /// <param name="msgType">Type of CAN-ISO-TP message (diagnostic or remote disgnostic message)</param>
        /// <param name="sourceAddr">Source address</param>
        /// <param name="targetAddr">Target address</param>
        /// <param name="targetType">Type of addressing (physical: -node to node-, or functional: -node to all-)</param>
        /// <param name="remoteAddr">Remote address (used only with remote disgnostic message)</param>
        public MappingStatus(
            uint canId = 0x00,
            uint canIdResponse = 0x00,
            TPCANTPIdType canIdType = TPCANTPIdType.PCANTP_ID_CAN_11BIT,
            TPCANTPFormatType formatType = TPCANTPFormatType.PCANTP_FORMAT_UNKNOWN,
            TPCANTPMessageType msgType = TPCANTPMessageType.PCANTP_MESSAGE_UNKNOWN,
            TPCANTPAddressingType targetType = TPCANTPAddressingType.PCANTP_ADDRESSING_UNKNOWN,
            byte sourceAddr = 0x00,
            byte targetAddr = 0x00,
            byte remoteAddr = 0x00)
        {
            m_canId = canId;
            m_canIdResponse = canIdResponse;
            m_canIdType = canIdType;
            m_formatType = formatType;
            m_msgType = msgType;
            m_sourceAddr = sourceAddr;
            m_targetAddr = targetAddr;
            m_targetType = targetType;
            m_remoteAddr = remoteAddr;
        }

        /// <summary>
        /// Gets a new instance of the object.
        /// </summary>
        /// <returns>The cloned instance of the object.</returns>
        public MappingStatus Clone()
        {
            return new MappingStatus(m_canId, m_canIdResponse, m_canIdType,
                m_formatType, m_msgType, m_targetType, m_sourceAddr, m_targetAddr, m_remoteAddr);
        }
        /// <summary>
        /// States if an ISO-TP mapping is NOT required for this configuration.
        ///  Some 29 bits CAN ID ISO-TP message with specific format addressing 
        ///  do NOT require ISO-TP mappings.
        /// </summary>
        public bool IsAutomatic
        {
            get
            {
                if (m_canIdType == TPCANTPIdType.PCANTP_ID_CAN_29BIT && (
                    this.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_FIXED_NORMAL ||
                    this.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_MIXED ||
                    this.m_formatType == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                    )
                    return true;
                return false;
            }
        }

        /// <summary>
        /// The formatted name of this mapping configuration.
        /// </summary>
        public string Name
        {
            get
            {
                String name;

                name = CanIdType + ", " + FormatType + ", " + TargetType;
                if (!IsAutomatic)
                    name += ": " + SourceAddress + " -> " + TargetAddress;
                return name;
            }

        }

        /// <summary>
        /// The CAN ID of the configured mapping as a string.
        /// Note: automatic mapping has no CAN ID, since it is 
        /// handled on the fly by the ISO-TP API.
        /// </summary>
        public string CanId
        {
            get
            {
                if (IsAutomatic)
                    return "-";
                return CanTpUtils.GetCanId(m_canId, m_canIdType == TPCANTPIdType.PCANTP_ID_CAN_29BIT);
            }
        }
        /// <summary>
        /// The CAN ID response of the configured mapping as a string.
        /// Note: automatic mapping has no CAN ID response, since it is 
        /// handled on the fly by the ISO-TP API.
        /// </summary>
        public string CanIdResponse
        {
            get
            {
                if (IsAutomatic)
                    return "-";
                return CanTpUtils.GetCanId(m_canIdResponse, m_canIdType == TPCANTPIdType.PCANTP_ID_CAN_29BIT);
            }
        }
        /// <summary>
        /// The CAN ID type of the configured mapping as a string.
        /// </summary>
        public string CanIdType
        {
            get { return CanTpUtils.GetCanIdType(m_canIdType); }
        }
        /// <summary>
        /// The ISO-TP message type of the configured mapping as a string.
        /// </summary>
        public string MsgType
        {
            get { return CanTpUtils.GetMsgType(m_msgType); }
        }
        /// <summary>
        /// The ISO-TP addressing format type of the configured mapping as a string.
        /// </summary>
        public string FormatType
        {
            get { return CanTpUtils.GetFormatType(m_formatType); }
        }
        /// <summary>
        /// The ISO-TP target address type ID type of the configured mapping as a string.
        /// </summary>
        public string TargetType
        {
            get { return CanTpUtils.GetTargetType(m_targetType); }
        }
        /// <summary>
        /// The source address of the configured mapping as a string.
        /// </summary>
        public string SourceAddress
        {
            get
            {
                if (IsAutomatic)
                    return "-";
                return CanTpUtils.GetAddress(m_sourceAddr);
            }
        }
        /// <summary>
        /// The target address of the configured mapping as a string.
        /// </summary>
        public string TargetAddress
        {
            get
            {
                if (IsAutomatic)
                    return "-";
                return CanTpUtils.GetAddress(m_targetAddr);
            }
        }
        /// <summary>
        /// The remote address of the configured mapping as a string.
        /// </summary>
        public string RemoteAddress
        {
            get
            {
                if (IsAutomatic)
                    return "-";
                return CanTpUtils.GetAddress(m_remoteAddr);
            }
        }
    }

    /// <summary>
    /// Class that stores information on a received ISO-TP message.
    /// </summary>
    public class MessageStatus
    {
        /// <summary>
        /// The last ISO-TP message received.
        /// </summary>
        private TPCANTPMsg m_msg;
        /// <summary>
        /// The timestamp of the last ISO-TP message received.
        /// </summary>
        private TPCANTPTimestamp m_timeStamp;
        /// <summary>
        /// The timestamp of the before last ISO-TP message received.
        /// </summary>
        private TPCANTPTimestamp m_timeStampOld;
        /// <summary>
        /// Index of the object in the message listView.
        /// </summary>
        private int m_index;
        /// <summary>
        /// Number of similar ISO-TP message received 
        /// (i.e. messages having the same Network Address Information).
        /// </summary>
        private int m_count;
        /// <summary>
        /// States if the timestamp should be displayed as a period.
        /// </summary>
        private bool m_showPeriod;
        /// <summary>
        /// States if the object has been modified but not updated/displayed on the UI.
        /// </summary>
        private bool m_wasChanged;

        /// <summary>
        /// Default constructor.
        /// </summary>
        /// <param name="canTpMsg">The received ISO-TP message.</param>
        /// <param name="canTpTimestamp">The timestamp when the message was received.</param>
        /// <param name="listIndex">Position of the messageStatus in the message listView.</param>
        public MessageStatus(TPCANTPMsg canTpMsg, TPCANTPTimestamp canTpTimestamp, int listIndex)
        {
            m_msg = canTpMsg;
            m_timeStamp = canTpTimestamp;
            m_timeStampOld = canTpTimestamp;
            m_index = listIndex;
            m_count = 1;
            m_showPeriod = true;
            m_wasChanged = false;
        }

        /// <summary>
        /// Updates the messageStatus' information with a newly received ISO-TP message.
        /// </summary>
        /// <param name="canTpMsg">The received ISO-TP message.</param>
        /// <param name="canTpTimestamp">The timestamp when the message was received.</param>
        public void update(TPCANTPMsg canTpMsg, TPCANTPTimestamp canTpTimestamp)
        {
            m_msg = canTpMsg;
            m_timeStampOld = m_timeStamp;
            m_timeStamp = canTpTimestamp;
            m_wasChanged = true;
            m_count += 1;
        }

        /// <summary>
        /// States if a message has the same network address information as 
        /// the last ISO-TP message received.
        /// </summary>
        /// <param name="canTpMsg">The ISO-TP message to compare to.</param>
        /// <returns></returns>
        public bool isSameNetAddrInfo(TPCANTPMsg canTpMsg)
        {
            return (m_msg.SA == canTpMsg.SA &&
                m_msg.TA == canTpMsg.TA &&
                m_msg.RA == canTpMsg.RA &&
                m_msg.IDTYPE == canTpMsg.IDTYPE &&
                m_msg.FORMAT == canTpMsg.FORMAT &&
                m_msg.TA_TYPE == canTpMsg.TA_TYPE &&
                m_msg.MSGTYPE == canTpMsg.MSGTYPE);
        }

        #region Getters / Setters
        /// <summary>
        /// The last ISO-TP message received.
        /// </summary>
        public TPCANTPMsg CanTpMsg
        {
            get { return m_msg; }
        }
        /// <summary>
        /// The timestamp of the last ISO-TP message received.
        /// </summary>
        public TPCANTPTimestamp Timestamp
        {
            get { return m_timeStamp; }
        }
        /// <summary>
        /// The index of the object in the "received message" listView.
        /// </summary>
        public int Position
        {
            get { return m_index; }
        }
        /// <summary>
        /// Number of time a similar message was received (same network address information).
        /// </summary>
        public int Count
        {
            get { return m_count; }
        }
        /// <summary>
        /// States if the timestamp should be displayed as a period.
        /// </summary>
        public bool ShowingPeriod
        {
            get { return m_showPeriod; }
            set
            {
                if (m_showPeriod ^ value)
                {
                    m_showPeriod = value;
                    m_wasChanged = true;
                }
            }
        }
        /// <summary>
        /// States if the object was modified since last display.
        /// </summary>
        public bool MarkedAsUpdated
        {
            get { return m_wasChanged; }
            set { m_wasChanged = value; }
        }

        /// <summary>
        /// Unique ID as a string of the last message received.
        /// </summary>
        public string UID
        {
            get { return CanTpUtils.GetUniqueId(m_msg); }
        }
        /// <summary>
        /// CAN ID type as a string of the last message received.
        /// </summary>
        public string CanIdType
        {
            get { return CanTpUtils.GetCanIdType(m_msg.IDTYPE); }
        }
        /// <summary>
        /// ISO-TP Message type as a string of the last message received.
        /// </summary>
        public string MsgType
        {
            get { return CanTpUtils.GetMsgType(m_msg.MSGTYPE); }
        }
        /// <summary>
        /// ISO-TP addressing format type as a string of the last message received.
        /// </summary>
        public string FormatType
        {
            get { return CanTpUtils.GetFormatType(m_msg.FORMAT); }
        }
        /// <summary>
        /// ISO-TP target addressing type as a string of the last message received.
        /// </summary>
        public string TargetType
        {
            get { return CanTpUtils.GetTargetType(m_msg.TA_TYPE); }
        }
        /// <summary>
        /// ISO-TP source address as a string of the last message received.
        /// </summary>
        public string SourceAddress
        {
            get
            {
                if (m_msg.FORMAT == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                    // Unique SA is 11bits: store bits [7..0] in SA
                    // and the rest in [3..0] of RA
                    return CanTpUtils.GetCanId(((uint)((m_msg.RA >> 4) & 0x7)) << 8 | m_msg.SA, false);
                return CanTpUtils.GetAddress(m_msg.SA);
            }
        }
        /// <summary>
        /// ISO-TP target address as a string of the last message received.
        /// </summary>
        public string TargetAddress
        {
            get
            {
                if (m_msg.FORMAT == TPCANTPFormatType.PCANTP_FORMAT_ENHANCED)
                    // Unique TA is 11bits: store bits [7..0] in TA
                    // and the rest in [7..4] of RA
                    return CanTpUtils.GetCanId(((uint)(m_msg.RA & 0x7)) << 8 | m_msg.TA, false);
                return CanTpUtils.GetAddress(m_msg.TA);
            }
        }
        /// <summary>
        /// ISO-TP remote address as a string of the last message received.
        /// </summary>
        public string RemoteAddress
        {
            get { return CanTpUtils.GetAddress(m_msg.RA); }
        }
        /// <summary>
        /// Lenght of the data as a string of the last message received.
        /// </summary>
        public string LengthString
        {
            get { return m_msg.LEN.ToString(); }
        }
        /// <summary>
        /// Data as a string of the last message received.
        /// </summary>
        public string DataString
        {
            get { return CanTpUtils.GetDataString(m_msg); }
        }
        /// <summary>
        /// ISO-TP Network result as a string of the last message received.
        /// </summary>
        public string ResultString
        {
            get { return CanTpUtils.GetResult(m_msg.RESULT); }
        }
        /// <summary>
        /// Timestamp (or period) as a string of the last message received.
        /// </summary>
        public string TimeString
        {
            get { return CanTpUtils.GetTimeString(m_timeStamp, m_showPeriod, m_timeStampOld); }
        }
        #endregion
    }
}
